/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_FORM_DATA;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_RELATED;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_X_MIXED_REPLACE;

public class DescriptorOperationMultipartResponseNotSupportedRule extends DescriptorValidationRule {

  public DescriptorOperationMultipartResponseNotSupportedRule() {
    super("Multipart response operations are not supported",
          "All operations declared must not return a multipart response",
          ERROR);
  }

  private static final List<String> multipartMediaTypes = new ArrayList<String>() {

    {
      add(MULTIPART_MIXED.toString());
      add(MULTIPART_FORM_DATA.toString());
      add(MULTIPART_RELATED.toString());
      add(MULTIPART_X_MIXED_REPLACE.toString());
    }
  };

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();
    validateEndpointsAPI(apiModel, validationResults);
    validateEndpointsDescriptor(connectorDescriptor, validationResults);
    validateOperationAdaptersDescriptor(connectorDescriptor, validationResults);
    return validationResults;
  }

  private void validateEndpointsAPI(APIModel apiModel, List<ValidationResult> validationResults) {
    apiModel.getOperationsModel().stream().forEach(operationModel -> {
      if (operationModel.getOutputMetadataModel() != null) {
        operationModel.getOutputMetadataModel().stream().forEach(apiTypeModel -> {
          if (apiTypeModel.getMediaType() != null) {
            if (multipartMediaTypes.stream()
                .anyMatch(multipartMediaType -> multipartMediaType.equals(apiTypeModel.getMediaType().toString()))) {
              validationResults.add(getValidationError(operationModel));
            }
          }
        });
      }
    });
  }

  private void validateEndpointsDescriptor(ConnectorDescriptor connectorDescriptor,
                                           List<ValidationResult> validationResults) {
    connectorDescriptor.getEndpoints().stream().forEach(endPointDescriptor -> endPointDescriptor.getOperations().stream()
        .filter(
                endPointOperationDescriptor -> multipartMediaTypes.stream()
                    .anyMatch(multipartMediaType -> multipartMediaType
                        .equals(endPointOperationDescriptor.getOutputMediaType())))
        .forEach(endPointDescriptorOperation -> validationResults
            .add(getValidationError(endPointDescriptorOperation, endPointDescriptor))));
  }

  private void validateOperationAdaptersDescriptor(ConnectorDescriptor connectorDescriptor,
                                                   List<ValidationResult> validationResults) {
    connectorDescriptor.getOperationAdapterDescriptors().stream().forEach(operationAdapterDescriptor -> {
      if (operationAdapterDescriptor.getResponseBindings() != null
          && operationAdapterDescriptor.getResponseBindings().getBodyBinding() != null &&
          operationAdapterDescriptor.getResponseBindings().getBodyBinding().getContentType() != null) {
        if (multipartMediaTypes.stream().anyMatch(multipartMediaType -> multipartMediaType.equals(
                                                                                                  operationAdapterDescriptor
                                                                                                      .getResponseBindings()
                                                                                                      .getBodyBinding()
                                                                                                      .getContentType()))) {
          validationResults.add(getValidationError(operationAdapterDescriptor));
        }
      }
    });
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor descriptorOperation) {
    String detail = "Operation adapter: "
        + descriptorOperation.getOperationId()
        + " returns a multipart response.";

    return new ValidationResult(this, detail, descriptorOperation.getLocation());
  }

  private ValidationResult getValidationError(OperationDescriptor descriptorOperation, EndPointDescriptor endPointDescriptor) {
    String detail = "Endpoint: "
        + descriptorOperation.getMethod() + " " + endPointDescriptor.getPath()
        + " returns a multipart response.";

    return new ValidationResult(this, detail, descriptorOperation.getLocation());
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String detail = "Endpoint in API spec file: "
        + apiOperationModel.getHttpMethod() + " " + apiOperationModel.getPath()
        + " returns a multipart response.";

    return new ValidationResult(this, detail, null);
  }
}
