/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.getOutputMediaType;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.mule.runtime.http.api.HttpHeaders.Values.BOUNDARY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.core.MediaType;

public class OperationAdapterBodyBindingShouldDefineBoundaryOnMultipartFormDataRule extends ConnectorModelValidationRule {

  public OperationAdapterBodyBindingShouldDefineBoundaryOnMultipartFormDataRule() {
    super("Body expression used in request binding for multipart endpoints should define the boundary parameter as part of the mime type declaration.",
          EMPTY, ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorModel connectorModel) {
    final List<ValidationResult> results = new ArrayList<>();

    List<ConnectorOperation> operations = connectorModel.getOperations();
    for (ConnectorOperation operation : operations) {
      Optional<List<ParameterBinding>> requestBindings = operation.getRequestBindings();
      if (requestBindings.isPresent()) {
        if (operation.getInputMetadata() != null
            && operation.getInputMetadata().getMediaType().equals(MULTIPART_FORM_DATA_TYPE)) {
          Optional<ParameterBinding> bodyParameterBinding = requestBindings.get().stream()
              .filter(parameterBinding -> parameterBinding.getParameterType().equals(ParameterType.BODY)).findFirst();
          if (bodyParameterBinding.isPresent()) {
            Optional<MediaType> outputMediaType = getOutputMediaType(bodyParameterBinding.get().getExpression());
            if (outputMediaType.isPresent() &&
                outputMediaType.get().isCompatible(MULTIPART_FORM_DATA_TYPE) &&
                !outputMediaType.get().getParameters().containsKey(BOUNDARY)) {
              results.add(getValidationError(operation));
            }
          }
        }
      }
    }

    return results;
  }

  private ValidationResult getValidationError(ConnectorOperation connectorOperation) {
    String detail = format("Body expression binding defined in Operation [%s]"
        + " has an expression which output is a multipart/form-data but it has to define the boundary parameter"
        + " as part of output directive (e.g., output multipart/form-data boundary='__c2341').",
                           connectorOperation.getOperationIdentifier());

    return new ValidationResult(this, detail, null);
  }
}
