/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.BODY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.util.Collections.EMPTY_LIST;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.join;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

public class ParametersUsedInOperationAdapterBindingsShouldExistRule extends DescriptorValidationRule {

  public ParametersUsedInOperationAdapterBindingsShouldExistRule() {
    // TODO: RSDK-826 Change WARN error level for ERROR when task is finished
    super("Parameter used in binding expression should exist.", EMPTY, WARN);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    List<OperationAdapterDescriptor> operationAdapterDescriptors = connectorDescriptor.getOperationAdapterDescriptors();
    for (OperationAdapterDescriptor operationAdapterDescriptor : operationAdapterDescriptors) {
      AuxiliarParameterRequestBindingsDescriptor requestBindings = operationAdapterDescriptor.getRequestBindings();
      if (requestBindings != null) {
        List<String> parametersAvailable = operationAdapterDescriptor.getParameters().stream()
            .map(AuxiliarParameterDescriptor::getName).collect(Collectors.toList());
        analyzeParameters(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getHeaders(), HEADER);
        analyzeParameters(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getQueryParameters(), QUERY);
        analyzeParameters(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getUriParameters(), URI);
        analyzeBody(results, operationAdapterDescriptor, parametersAvailable, requestBindings.getWeaveExpression());
      }
    }

    return results;
  }

  private void analyzeParameters(List<ValidationResult> results, OperationAdapterDescriptor operationAdapterDescriptor,
                                 List<String> parametersAvailable, List<AuxiliarParameterBindingDescriptor> bindings,
                                 ParameterType bindingParameterType) {

    for (AuxiliarParameterBindingDescriptor binding : bindings) {

      List<String> missingParameters = isValidBinding(binding.getValue(), parametersAvailable);
      if (!missingParameters.isEmpty()) {
        results.add(getValidationError(operationAdapterDescriptor, binding.getName(), bindingParameterType, missingParameters));
      }
    }
  }

  private List<String> isValidBinding(String expression, List<String> parametersAvailable) {
    if (StringUtils.isBlank(expression)) {
      return EMPTY_LIST;
    }

    List<String> parametersErrors = new ArrayList<>();
    String[] parameters = DataWeaveExpressionParser.selectionsFromBinding(expression, ParameterType.AUXILIAR.getBinding());

    for (String parameter : parameters) {
      if (!parametersAvailable.stream().anyMatch(x -> x.equals(parameter))) {
        parametersErrors.add(parameter);
      }
    }

    return parametersErrors;
  }

  private void analyzeBody(List<ValidationResult> results, OperationAdapterDescriptor operationAdapterDescriptor,
                           List<String> parametersAvailable, String bodyExpression) {

    List<String> missingParameters = isValidBinding(bodyExpression, parametersAvailable);
    if (!missingParameters.isEmpty()) {
      results.add(getValidationError(operationAdapterDescriptor, BODY.getName(), BODY, missingParameters));
    }
  }

  private ValidationResult getValidationError(OperationAdapterDescriptor operationAdapterDescriptor,
                                              String binding,
                                              ParameterType parameterType,
                                              List<String> missingParameters) {
    String detail = parameterType.getName()
        + " parameter binding '"
        + binding
        + "' defined in Operation with name: "
        + operationAdapterDescriptor.getOperationId()
        + " has an expression with the following invalid parameters: "
        + join(missingParameters, ",");

    return new ValidationResult(this, detail, operationAdapterDescriptor.getLocation());
  }
}
