/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.getValueProviderBodyLevelReferenceDescriptors;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.common.ArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverArgumentDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.valueprovider.ValueProviderDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class ValueProviderBodyLevelReferenceArgumentsHaveValidFormatRule extends DescriptorValidationRule {

  private static final String ERROR_TEMPLATE = "Value '%s' for argument '%s' does not match the expected format.";

  public ValueProviderBodyLevelReferenceArgumentsHaveValidFormatRule() {
    super("Value Provider at body level arguments must respect the valid format (i.e. uriParameters.X, queryParameters.X, headers.X or body.X)",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> validationResults = new ArrayList<>();

    for (ResolverReferenceDescriptor<ValueProviderDefinitionDescriptor> reference : getValueProviderBodyLevelReferenceDescriptors(connectorDescriptor)) {
      final List<ResolverArgumentDescriptor> arguments = reference.getArguments();

      final List<ValidationResult> errors = arguments.stream()
          .filter(x -> !isValidArgumentValueFormat(x.getValue().getExpression()))
          .map(this::getValidationError)
          .collect(toList());

      validationResults.addAll(errors);
    }

    return validationResults;
  }

  private static String[] BINDINGS = new String[] {
      ParameterType.URI.getBinding(),
      ParameterType.QUERY.getBinding(),
      ParameterType.HEADER.getBinding(),
      ParameterType.AUXILIAR.getBinding(),
      ParameterType.BODY.getBinding(),
      ParameterType.CONFIGURATION.getBinding(),
      ParameterType.CONNECTION.getBinding()
  };

  private boolean isValidArgumentValueFormat(String value) {
    if (isBlank(value)) {
      return false;
    }
    try {
      ExpressionHandlerUtils.compileDataWeaveScript(value, BINDINGS);
      return true;
    } catch (CompilationException e) {
      return false;
    }
  }

  private ValidationResult getValidationError(ArgumentDescriptor argument) {
    return new ValidationResult(this,
                                format(ERROR_TEMPLATE, argument.getValue(), argument.getName()),
                                argument.getLocation());
  }
}
