/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.Position;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule;

public class ValidationResult {

  private final ValidationRule rule;
  private final String detail;
  private final Location location;

  public ValidationResult(ValidationRule rule, String detail) {
    this(rule, detail, DescriptorElementLocation.EMPTY);
  }

  public ValidationResult(ValidationRule rule, String detail, DescriptorElementLocation descriptorElementLocation) {
    this.rule = rule;
    this.detail = detail;
    if (descriptorElementLocation != null) {
      this.location = new Location(descriptorElementLocation);
    } else {
      this.location = null;
    }
  }

  public ValidationRule getRule() {
    return rule;
  }

  public ValidationRule.Level getLevel() {
    return rule.getLevel();
  }

  public String getDetail() {
    return detail;
  }

  public Location getLocation() {
    return location;
  }

  @Override
  public String toString() {
    StringBuilder stringBuilder =
        new StringBuilder(format("[%s] Rule: %s", rule.getLevel().name(), rule.getDescription()));

    stringBuilder.append(lineSeparator());

    if (isNotBlank(rule.getExplanation())) {
      stringBuilder
          .append("Explanation: ")
          .append(rule.getExplanation())
          .append(lineSeparator());
    }

    if (isNotBlank(detail)) {
      stringBuilder
          .append("Detail: ")
          .append(detail)
          .append(lineSeparator());
    }

    stringBuilder
        .append("Location: ")
        .append(location)
        .append(lineSeparator());

    return stringBuilder.toString();
  }

  public static class Location {

    private final boolean locatedInDescriptor;
    private final String file;
    private final Position start, end;

    public Location(DescriptorElementLocation descriptorLocation) {
      this.locatedInDescriptor = !descriptorLocation.isEmpty();
      this.file = descriptorLocation.getFile();
      this.start = descriptorLocation.getStart();
      this.end = descriptorLocation.getEnd();
    }

    public String toString() {
      if (!locatedInDescriptor) {
        return "File: API Specification";
      }
      int endLine = end.getLine();
      int startLine = start.getLine();
      if (end.getColumn() == 1)
        endLine--;
      StringBuilder sb = new StringBuilder().append("File: ").append(file);
      if (startLine == endLine)
        sb.append(" (line ").append(startLine);
      else
        sb.append(" (lines ").append(startLine).append(" to ").append(endLine);
      sb.append(")");
      return sb.toString();
    }

    public String getFile() {
      return file;
    }

    public Position getStart() {
      return start;
    }

    public Position getEnd() {
      return end;
    }
  }
}
