/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.complete;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.GET;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.OPTIONS;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.PATCH;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.POST;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.CompleteConnectorModelBuilderValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;

import java.util.List;

import org.apache.commons.lang3.StringUtils;

public class OperationMustDeclareResponseBodyRule extends CompleteConnectorModelBuilderValidationRule {

  // R003
  public OperationMustDeclareResponseBodyRule() {
    super("Operations must declare a response body",
          "GET, POST, PATCH and OPTIONS operations defined in the API spec must declare a response body and a type for that body. If they don't the connector descriptor must skip this check, define a response body for the operation, or declare it as void.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModelBuilder connectorModelBuilder) {
    return connectorModelBuilder.getOperationBuilders().stream()
        .filter(op -> op.isIgnored() == null || !op.isIgnored())
        .filter(op ->
        // TODO RSDK-805: fix this when tackling the issue
        StringUtils.isBlank(op.getFqn()))
        .filter(op -> op.getIsVoidOperation() == null || !op.getIsVoidOperation())
        .filter(op -> !skippedOutputTypeValidation(op, connectorModelBuilder))
        .filter(op -> httpMethodApplies(op.getMethod()))
        .filter(op -> !hasOutputTypeDefinition(op))
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean hasOutputTypeDefinition(OperationBuilder operationBuilder) {
    if (operationBuilder.hasForcedOutputTypeSchema()) {
      return true;
    }

    return operationBuilder.getOutputMetadataBuilders().values().stream().anyMatch(x -> !x.isEmpty());
  }

  private boolean skippedOutputTypeValidation(OperationBuilder operationBuilder, ConnectorModelBuilder connectorModelBuilder) {
    Boolean skipped = defaultIfNull(operationBuilder.getSkipOutputTypeValidation(),
                                    connectorModelBuilder.getSkipOutputTypeValidation());

    return skipped != null && skipped;
  }

  private boolean httpMethodApplies(String httpMethod) {
    return httpMethod.equalsIgnoreCase(GET.name())
        || httpMethod.equalsIgnoreCase(POST.name())
        || httpMethod.equalsIgnoreCase(PATCH.name())
        || httpMethod.equalsIgnoreCase(OPTIONS.name());
  }

  private ValidationResult getValidationError(OperationBuilder operationBuilder) {
    String detail =
        format("Operation with PATH: %s and METHOD: %s does not declare a response body. There must be at least a successful status code response with a response body specified.",
               operationBuilder.getPath(),
               operationBuilder.getMethod().toUpperCase());

    return new ValidationResult(this, detail);
  }
}
