/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.connectormodel.builder.step;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.lang.String.join;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TriggerBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.StepConnectorModelBuilderValidationRule;

import java.util.ArrayList;
import java.util.List;

public class TriggerMandatoryFieldsRule extends StepConnectorModelBuilderValidationRule {

  public TriggerMandatoryFieldsRule() {
    super("Trigger declarations must provide all mandatory fields.",
          "Path, method, displayName and items are mandatory fields for trigger declarations.",
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(ConnectorModelBuilder connectorModelBuilder) {
    List<ValidationResult> results = new ArrayList<>();

    connectorModelBuilder.getTriggerBuilders().stream()
        .filter(trigger -> isBlank(trigger.getFqn()))
        .forEach(trigger -> validateMandatoryFields(results, trigger));

    return results;
  }

  private void validateMandatoryFields(List<ValidationResult> results, TriggerBuilder triggerBuilder) {
    boolean hasPath = isNotBlank(triggerBuilder.getPath());
    boolean hasMethod = triggerBuilder.getMethod() != null;
    boolean hasDisplayName = isNotBlank(triggerBuilder.getDisplayName());
    boolean hasItems = isNotBlank(triggerBuilder.getItemsExpression());

    if (!hasPath || !hasMethod || !hasDisplayName || !hasItems) {
      results
          .add(getValidationError(triggerBuilder.getName(), !hasPath, !hasMethod, !hasDisplayName, !hasItems));
    }
  }

  final static String ERROR_MESSAGE_TEMPLATE = "The trigger declaration '%s' is missing some mandatory fields: %s.";

  private ValidationResult getValidationError(String name, boolean missingPath, boolean missingMethod, boolean missingDisplayName,
                                              boolean missingItemsExpression) {
    List<String> missingFields = new ArrayList<>();

    if (missingPath) {
      missingFields.add("Path");
    }

    if (missingMethod) {
      missingFields.add("Method");
    }

    if (missingDisplayName) {
      missingFields.add("DisplayName");
    }

    if (missingItemsExpression) {
      missingFields.add("Items");
    }

    return new ValidationResult(this,
                                format(ERROR_MESSAGE_TEMPLATE, name, join(", ", missingFields)));
  }
}
