/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.util.Arrays.stream;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarBodyBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterResponseBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverrideResolverDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesBodyDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public abstract class AbstractCustomFieldRule extends DescriptorValidationRule {

  protected static final Map<String, String> BINDINGS_MAPPING = new HashMap<String, String>() {

    {
      put(DWBindings.QUERY.getBinding(), ParameterType.QUERY.getBinding());
      put(DWBindings.PATH.getBinding(), ParameterType.URI.getBinding());
      put(DWBindings.HEADER.getBinding(), ParameterType.HEADER.getBinding());
    }
  };

  protected static final String[] BINDINGS = new String[] {
      DWBindings.REQUEST.getBinding(),
      ParameterType.URI.getBinding(),
      ParameterType.QUERY.getBinding(),
      ParameterType.HEADER.getBinding()
  };

  public AbstractCustomFieldRule(String description) {
    super(description, EMPTY, ERROR);
  }

  protected abstract void validateOverrideResolver(OperationAdapterDescriptor operationAdapterDescriptor,
                                                   OverrideResolverDescriptor overrideResolver,
                                                   List<ValidationResult> results);

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    final List<ValidationResult> results = new ArrayList<>();

    for (OperationAdapterDescriptor operationAdapterDescriptor : connectorDescriptor.getOperationAdapterDescriptors()) {
      validateRequestBodyCustomFields(operationAdapterDescriptor, results);
      validateResponseBodyCustomFields(operationAdapterDescriptor, results);
    }

    return results;
  }

  private void validateRequestBodyCustomFields(OperationAdapterDescriptor operationAdapterDescriptor,
                                               List<ValidationResult> results) {
    if (operationAdapterDescriptor.getBaseEndpoint() != null
        && operationAdapterDescriptor.getBaseEndpoint().getOverrides() != null) {
      OverridesBodyDescriptor overridesBody = operationAdapterDescriptor.getBaseEndpoint().getOverrides().getBody();
      if (overridesBody != null) {
        if (overridesBody.getOverrideResolver() != null) {
          validateOverrideResolver(operationAdapterDescriptor, overridesBody.getOverrideResolver(), results);
        }
        overridesBody.getFieldDescriptors().stream()
            .filter(field -> field.getOverrideResolver() != null)
            .forEach(field -> validateOverrideResolver(operationAdapterDescriptor, field.getOverrideResolver(), results));
      }
    }
  }

  private void validateResponseBodyCustomFields(OperationAdapterDescriptor operationAdapterDescriptor,
                                                List<ValidationResult> results) {
    AuxiliarParameterResponseBindingsDescriptor responseBindings = operationAdapterDescriptor.getResponseBindings();
    if (responseBindings != null) {
      AuxiliarBodyBindingDescriptor overridesBody = responseBindings.getBodyBinding();
      if (overridesBody != null) {
        if (overridesBody.getOverrideResolver() != null) {
          validateOverrideResolver(operationAdapterDescriptor, overridesBody.getOverrideResolver(), results);
        }
        overridesBody.getAuxiliarBodyField().stream()
            .filter(field -> field.getOverrideResolver() != null)
            .forEach(field -> validateOverrideResolver(operationAdapterDescriptor, field.getOverrideResolver(), results));
      }
    }
  }

  protected String getParameterTypeReference(String value) {
    if (isBlank(value)) {
      return null;
    }
    return stream(BINDINGS)
        .filter(accessorName -> DataWeaveExpressionParser.isBindingUsed(value, accessorName))
        .findFirst()
        .orElse(null);
  }

  protected String getParameterReference(String value, List<String> parameterType) {
    if (isBlank(value)) {
      return null;
    }
    return stream(BINDINGS)
        .flatMap((b) -> stream(DataWeaveExpressionParser.selectionsFromBinding(value, b, parameterType)))
        .findFirst()
        .orElse(null);
  }


}
