/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.selectionsFromBinding;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.BaseParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ConnectorModelValidationRule;

import java.util.List;
import java.util.Map;

public abstract class AbstractEnricherArgumentsOptionalityValidationRule extends ConnectorModelValidationRule {

  protected static final String PARAMETERS = "parameters";

  protected static final String QUERY_PARAMETERS = "queryParameters";
  protected static final String URI_PARAMETERS = "uriParameters";
  protected static final String HEADERS = "headers";

  public AbstractEnricherArgumentsOptionalityValidationRule(String enricherType) {
    super(format("When sending an argument to a %s if the value expression is based on non required parameters the resolver may not be invoked",
                 enricherType),
          EMPTY, WARN);
  }

  protected void validateParameters(List<ValidationResult> validationResults, ConnectorOperation connectorOperation,
                                    Map<String, Object> context, List<Argument> arguments) {
    if (connectorOperation.getParameters().isPresent()) {
      doValidation(validationResults, connectorOperation, arguments,
                   connectorOperation.getParameters().get().stream()
                       .collect(toList()),
                   PARAMETERS, context);
    }
    doValidation(validationResults, connectorOperation, arguments,
                 connectorOperation.getQueryParameters().stream()
                     .map(declaredParameters -> (BaseParameter) declaredParameters)
                     .collect(toList()),
                 QUERY_PARAMETERS, context);
    doValidation(validationResults, connectorOperation, arguments,
                 connectorOperation.getUriParameters().stream()
                     .map(declaredParameters -> (BaseParameter) declaredParameters)
                     .collect(toList()),
                 URI_PARAMETERS, context);
    doValidation(validationResults, connectorOperation, arguments,
                 connectorOperation.getHeaders().stream()
                     .map(declaredParameters -> (BaseParameter) declaredParameters)
                     .collect(toList()),
                 HEADERS, context);
  }

  private void doValidation(List<ValidationResult> validationResults,
                            ConnectorOperation connectorOperation,
                            List<Argument> arguments,
                            List<BaseParameter> parameters,
                            String variableName,
                            Map<String, Object> context) {
    for (Argument argument : arguments) {
      String[] referencedParameters = selectionsFromBinding(argument.getValue().getValue(), variableName);
      for (int i = 0; i < referencedParameters.length; i++) {
        String referencedParameter = referencedParameters[i];
        parameters.stream()
            .filter(parameter -> parameter.getExternalName().equals(referencedParameter))
            .filter(parameter -> !parameter.isRequired())
            // Only a parameter has a default value!
            .filter(parameter -> ((parameter instanceof Parameter) && isBlank(((Parameter) parameter).getDefaultValue())) ||
                !(parameter instanceof Parameter))
            .findFirst()
            .map(parameter -> validationResults
                .add(getValidationError(connectorOperation, argument, variableName, parameter, context)));
      }
    }
  }

  protected abstract ValidationResult getValidationError(ConnectorOperation connectorOperation, Argument argument,
                                                         String parameterKind, BaseParameter parameter,
                                                         Map<String, Object> context);

}
