/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;

import java.util.ArrayList;
import java.util.List;

public class DescriptorAdditionalParameterMandatoryFieldsRule extends DescriptorValidationRule {

  public DescriptorAdditionalParameterMandatoryFieldsRule() {
    super("Additional security parameters declared in the connector descriptor must have mandatory fields",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> results = new ArrayList<>();

    for (SecuritySchemeBaseDescriptor securitySchemeBaseDescriptor : connectorDescriptor.getSecurity()) {
      if (securitySchemeBaseDescriptor.getQueryParameters() != null) {
        validateParametersExists(securitySchemeBaseDescriptor.getQueryParameters(),
                                 securitySchemeBaseDescriptor.getName(), results);
      }

      if (securitySchemeBaseDescriptor.getHeaders() != null) {
        validateParametersExists(securitySchemeBaseDescriptor.getHeaders(),
                                 securitySchemeBaseDescriptor.getName(), results);
      }
    }

    return results;
  }

  private void validateParametersExists(List<ParameterDescriptor> parameters,
                                        String securityScheme,
                                        List<ValidationResult> validationResults) {
    parameters.stream()
        .filter(ParameterDescriptor::isAdditional)
        .forEach(paramDesc -> validateParameterFields(paramDesc, securityScheme, validationResults));
  }

  private void validateParameterFields(ParameterDescriptor parameter,
                                       String securityScheme,
                                       List<ValidationResult> validationResults) {
    if (parameter.getDataType() == null) {
      validationResults.add(getValidationError(parameter, securityScheme, "type"));
    }
    if (isBlank(parameter.getDisplayName())) {
      validationResults.add(getValidationError(parameter, securityScheme, "displayName"));
    }
  }

  private ValidationResult getValidationError(ParameterDescriptor paramDescriptor, String securityScheme, String field) {
    String detail = format("Field '%s' from parameter '%s' in Security scheme '%s' is missing", field,
                           paramDescriptor.getParamName(), securityScheme);

    return new ValidationResult(this, detail, paramDescriptor.getLocation());
  }
}
