/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_FORM_DATA;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_RELATED;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_X_MIXED_REPLACE;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.CompleteConnectorModelBuilderValidationRule;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class DescriptorOperationMultipartResponseNotSupportedRule extends CompleteConnectorModelBuilderValidationRule {

  public DescriptorOperationMultipartResponseNotSupportedRule() {
    super("Multipart response operations are not supported",
          "All operations declared must not return a multipart response",
          ERROR);
  }

  private static final List<String> multipartMediaTypes = new ArrayList<String>() {

    {
      add(MULTIPART_MIXED.toString());
      add(MULTIPART_FORM_DATA.toString());
      add(MULTIPART_RELATED.toString());
      add(MULTIPART_X_MIXED_REPLACE.toString());
    }
  };

  @Override
  public List<ValidationResult> validate(ConnectorModelBuilder connectorModelBuilder) {
    List<ValidationResult> validationResults = new ArrayList<>();
    connectorModelBuilder.getOperationBuilders()
        .stream()
        .filter(operationBuilder -> operationBuilder.isIgnored() != null && !operationBuilder.isIgnored())
        .filter(this::hasMultipartResponse)
        .forEach(operationBuilder -> getValidationError(operationBuilder, validationResults));
    return validationResults;
  }

  private boolean hasMultipartResponse(OperationBuilder operationBuilder) {
    return multipartMediaTypes.stream().anyMatch(multipartMediaType -> operationBuilder.getOutputMetadataBuilders()
        .containsKey(MediaType.valueOf(multipartMediaType)));
  }

  private void getValidationError(OperationBuilder operationBuilder, List<ValidationResult> validationResults) {
    String detail;
    if (operationBuilder.isAdapter()) {
      detail = "Operation adapter: "
          + operationBuilder.getDescriptorIdentifier()
          + " returns a multipart response.";
    } else {
      detail = "Endpoint: "
          + operationBuilder.getMethod() + " " + operationBuilder.getPath()
          + " returns a multipart response.";
    }

    validationResults.add(new ValidationResult(this, detail));
  }

}
