/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseEndpointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OverridesRequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public class OverrideParameterDescriptorMustBePresentInApiRule extends DescriptorValidationRule {

  public OverrideParameterDescriptorMustBePresentInApiRule() {
    super("Operation Override parameters declared in the connector descriptor must be present in the API spec",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> results = new ArrayList<>();

    for (OperationAdapterDescriptor operationAdapterDescriptor : connectorDescriptor.getOperationAdapterDescriptors()) {
      if (operationAdapterDescriptor.getBaseEndpoint() != null &&
          operationAdapterDescriptor.getBaseEndpoint().getOverrides() != null) {
        BaseEndpointDescriptor baseEndpointDescriptor = operationAdapterDescriptor.getBaseEndpoint();
        OverridesRequestDescriptor overridesRequestDescriptor = baseEndpointDescriptor.getOverrides();

        Optional<APIOperationModel> apiOperationModel = apiModel.getOperationsModel().stream()
            .filter(aom -> matchesOperationIdOrMethodAndPath(aom, baseEndpointDescriptor))
            .findFirst();
        if (apiOperationModel.isPresent()) {
          results.addAll(validateOverridesParametersExists(overridesRequestDescriptor.getUriParameter(), URI,
                                                           apiOperationModel.get(), operationAdapterDescriptor));
          results.addAll(validateOverridesParametersExists(overridesRequestDescriptor.getQueryParameter(), QUERY,
                                                           apiOperationModel.get(), operationAdapterDescriptor));
          results.addAll(validateOverridesParametersExists(overridesRequestDescriptor.getHeader(), HEADER,
                                                           apiOperationModel.get(), operationAdapterDescriptor));
        }
      }
    }


    return results;
  }

  private boolean matchesOperationIdOrMethodAndPath(APIOperationModel apiOperationModel,
                                                    BaseEndpointDescriptor endpointDescriptor) {
    boolean matches = false;
    if (endpointDescriptor.getOperationStringIdentifier() != null
        && endpointDescriptor.getOperationStringIdentifier().equals(apiOperationModel.getOperationId())) {
      matches = true;
    } else {
      String methodAndPath = format("%s-%s", apiOperationModel.getHttpMethod(), apiOperationModel.getPath());
      if (methodAndPath.equalsIgnoreCase(endpointDescriptor.getOperationStringIdentifier())) {
        matches = true;
      }
    }
    return matches;
  }

  private List<ValidationResult> validateOverridesParametersExists(List<OverridesParameterDescriptor> param,
                                                                   ParameterType parameterType,
                                                                   APIOperationModel apiOperationModel,
                                                                   OperationAdapterDescriptor operationAdapterDescriptor) {
    return param.stream()
        .map(p -> validateParameterExists(p, parameterType, apiOperationModel, operationAdapterDescriptor))
        .filter(Objects::nonNull)
        .collect(toList());

  }


  private ValidationResult validateParameterExists(OverridesParameterDescriptor parameter,
                                                   ParameterType parameterType,
                                                   APIOperationModel apiOperationModel,
                                                   OperationAdapterDescriptor operationAdapterDescriptor) {

    return validateParameterExists(apiOperationModel, parameter, parameterType, operationAdapterDescriptor);
  }

  private ValidationResult validateParameterExists(APIOperationModel apiOperation,
                                                   OverridesParameterDescriptor parameter,
                                                   ParameterType parameterType,
                                                   OperationAdapterDescriptor operationAdapterDescriptor) {
    return apiOperation.getParameter(parameterType.getApiParameterType(), parameter.getParamName()).isPresent()
        ? null
        : getValidationError(parameter, parameterType, operationAdapterDescriptor);
  }

  private ValidationResult getValidationError(OverridesParameterDescriptor paramDescriptor,
                                              ParameterType parameterType,
                                              OperationAdapterDescriptor operationAdapterDescriptor) {
    String detail =
        "Operation defined with name: "
            + operationAdapterDescriptor.getOperationId() +
            " declares a "
            + parameterType.getName()
            + " named '"
            + paramDescriptor.getParamName() + "'" +
            " that couldn't be found in the API spec";

    return new ValidationResult(this, detail, paramDescriptor.getLocation());
  }
}
