/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.ArrayList;
import java.util.List;

public class RequiredParametersMustNotBeIgnoredRule extends DescriptorValidationRule {

  public RequiredParametersMustNotBeIgnoredRule() {
    super("Required operation parameters must not be ignored.",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {

    List<ValidationResult> results = new ArrayList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParametersAreRequiredAndIgnored(operationDescriptor.getExpects().getUriParameter(), URI,
                                                                 endPointDescriptor.getPath(), operationDescriptor.getMethod(),
                                                                 apiModel));

          results.addAll(validateParametersAreRequiredAndIgnored(operationDescriptor.getExpects().getQueryParameter(), QUERY,
                                                                 endPointDescriptor.getPath(), operationDescriptor.getMethod(),
                                                                 apiModel));

          results.addAll(validateParametersAreRequiredAndIgnored(operationDescriptor.getExpects().getHeader(), HEADER,
                                                                 endPointDescriptor.getPath(), operationDescriptor.getMethod(),
                                                                 apiModel));

        }
      }
    }

    return results;
  }

  private List<ValidationResult> validateParametersAreRequiredAndIgnored(List<ParameterDescriptor> parameters,
                                                                         ParameterType parameterType,
                                                                         String path,
                                                                         String method,
                                                                         APIModel apiModel) {
    return parameters.stream()
        .filter(parameterDescriptor -> parameterDescriptor.isIgnored() != null && parameterDescriptor.isIgnored())
        .filter(parameterDescriptor -> parameterIsRequired(parameterDescriptor, parameterType, path, method, apiModel))
        .map(parameterDescriptor -> getValidationError(path, method, parameterDescriptor, parameterType))
        .collect(toList());
  }

  private boolean parameterIsRequired(ParameterDescriptor parameter,
                                      ParameterType parameterType,
                                      String path,
                                      String method,
                                      APIModel apiModel) {

    APIOperationModel apiOperation = apiModel.findOperation(path, method).orElse(null);

    if (apiOperation == null) {
      return false;
    }

    APIParameterModel apiParameter =
        apiOperation.getParameter(parameterType.getApiParameterType(), parameter.getParamName()).orElse(null);
    if (apiParameter == null) {
      return false;
    }

    if (URI.equals(parameterType)) {
      return true;
    } else {
      return apiParameter.isRequired();
    }
  }

  private ValidationResult getValidationError(String path, String method, ParameterDescriptor paramDescriptor,
                                              ParameterType parameterType) {

    String detail = parameterType.getName()
        + " '"
        + paramDescriptor.getParamName()
        + "' defined in Operation with path: "
        + path
        + " and method: "
        + method
        + " cannot be ignored as it's required.";

    return new ValidationResult(this, detail, paramDescriptor.getLocation());
  }
}
