/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.descriptor;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ExpressionHandlerUtils;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TestConnectionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TestConnectionResponseValidationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorElementLocation;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.WeaveExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.DescriptorValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import org.mule.weave.v2.parser.phase.CompilationException;

import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.ATTRIBUTES_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.ERROR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;

public class TestConnectionExpressionMustCompileRule extends DescriptorValidationRule {

  private static final String[] IMPLICIT_INPUTS = {PAYLOAD_VAR, ATTRIBUTES_VAR};

  public TestConnectionExpressionMustCompileRule() {
    super("Expression is invalid",
          EMPTY,
          ERROR);
  }

  private void validateErrorTemplateExpression(TestConnectionDescriptor testConnectionDescriptor,
                                               List<ValidationResult> validationResults) {
    if (testConnectionDescriptor.getStatusCodeValidation() != null) {
      if (testConnectionDescriptor.getStatusCodeValidation().getErrorTemplateExpression() != null) {
        validateExpression(testConnectionDescriptor.getStatusCodeValidation()
            .getErrorTemplateExpression()
            .getLocation(),
                           testConnectionDescriptor.getStatusCodeValidation()
                               .getErrorTemplateExpression(),
                           IMPLICIT_INPUTS, validationResults);
      }
    }
  }

  private void validateResponseValidationExpression(TestConnectionDescriptor testConnectionDescriptor,
                                                    List<ValidationResult> validationResults) {
    for (TestConnectionResponseValidationDescriptor testConnectionResponseValidationDescriptor : testConnectionDescriptor
        .getResponseValidation()) {

      if (testConnectionResponseValidationDescriptor.getValidationExpression() != null) {
        validateExpression(testConnectionResponseValidationDescriptor.getValidationExpression()
            .getLocation(),
                           testConnectionResponseValidationDescriptor.getValidationExpression(),
                           IMPLICIT_INPUTS, validationResults);
      }

      if (testConnectionResponseValidationDescriptor.getErrorTemplateExpression() != null) {
        validateExpression(testConnectionResponseValidationDescriptor.getErrorTemplateExpression()
            .getLocation(),
                           testConnectionResponseValidationDescriptor.getErrorTemplateExpression(),
                           IMPLICIT_INPUTS, validationResults);
      }
    }
  }

  @Override
  public List<ValidationResult> validate(APIModel apiModel, ConnectorDescriptor connectorDescriptor) {
    List<ValidationResult> validationResults = new ArrayList<>();

    for (SecuritySchemeBaseDescriptor security : connectorDescriptor.getSecurity()) {
      if (security instanceof SecuritySchemeDescriptor) {
        SecuritySchemeDescriptor securityCasted = (SecuritySchemeDescriptor) security;
        if (securityCasted.getTestConnection() != null) {
          validateErrorTemplateExpression(securityCasted.getTestConnection(), validationResults);
          validateResponseValidationExpression(securityCasted.getTestConnection(), validationResults);
        }
      }
    }

    if (connectorDescriptor.getTestConnection() != null) {
      validateErrorTemplateExpression(connectorDescriptor.getTestConnection(), validationResults);
      validateResponseValidationExpression(connectorDescriptor.getTestConnection(), validationResults);
    }
    return validationResults;
  }

  private void validateExpression(DescriptorElementLocation location,
                                  WeaveExpressionDescriptor expression, String[] implicitInputs,
                                  List<ValidationResult> validationResults) {
    try {

      ExpressionHandlerUtils.compileDataWeaveScript(expression.getExpression(), implicitInputs);
    } catch (CompilationException e) {
      validationResults.add(getValidationError(location, expression.getExpression(), e));
    }
  }

  private ValidationResult getValidationError(DescriptorElementLocation location,
                                              String expression, CompilationException e) {
    String detail = format("Expression '%s' is not valid, error: \n%s", expression, e.getMessage());
    return new ValidationResult(this, detail, location);
  }

}
