/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules.files;

import static com.mulesoft.connectivity.rest.sdk.internal.validation.rules.ValidationRule.Level.WARN;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils.VALIDATION_ERROR_MESSAGE_TEMPLATE;
import static java.lang.String.format;
import static java.nio.file.Files.walk;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.validation.rules.FilesValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.util.ValidationUtils;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import org.slf4j.Logger;

public class RefinementsDontGoInOverrideFolderRule extends FilesValidationRule {

  private static final Logger LOGGER = getLogger(RefinementsDontGoInOverrideFolderRule.class);

  public RefinementsDontGoInOverrideFolderRule() {
    super("Refinements must be located in the refinement folder (src/main/refinement)", EMPTY, WARN);
  }

  @Override
  public List<ValidationResult> validate(Path outputDir, RestSdkRunConfiguration runConfiguration) {
    if (!runConfiguration.regenerateMode()) {
      return emptyList();
    }

    try {
      Path overrideDir = outputDir.resolve(runConfiguration.getOverrideDir());

      if (overrideDir.toFile().exists()) {
        return walk(overrideDir)
            .filter(Files::isRegularFile)
            .filter(ValidationUtils::isRefinementFile)
            .map(this::getValidationResult)
            .collect(toList());
      }

    } catch (IOException e) {
      String message = format(VALIDATION_ERROR_MESSAGE_TEMPLATE, e.getMessage());
      LOGGER.error(message, e);
      return emptyList();
    }

    return emptyList();
  }

  private ValidationResult getValidationResult(Path file) {
    String detail = format("File '%s' is a refinement and must be located in the 'refinement' folder", file.toString());

    return new ValidationResult(this, detail, null);
  }
}
