package com.mulesoft.extension.mq.internal;

import static com.mulesoft.extension.mq.internal.config.SubscriberAckMode.IMMEDIATE;
import static com.mulesoft.mq.restclient.internal.ExecutorUtils.stopExecutorService;

import com.mulesoft.extension.mq.api.message.AnypointMQMessageContext;
import com.mulesoft.extension.mq.api.message.MessageContextFactory;
import com.mulesoft.extension.mq.internal.config.SubscriberConfiguration;
import com.mulesoft.extension.mq.internal.domain.MessageListener;
import com.mulesoft.mq.restclient.api.AnypointMqMessage;
import com.mulesoft.mq.restclient.api.Destination;
import com.mulesoft.mq.restclient.internal.AbstractCourierRestClient;

import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractSubscriber {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSubscriber.class);
    public static final long DEFAULT_POLLING_TIME = 1000;
    public static final String DEFAULT_POLLING_TIME_STRING = "1000";

    private final MessageListener messageListener;
    private final MessageContextFactory messageContextFactory;
    protected final Destination destination;
    protected final SubscriberConfiguration subscriberConfiguration;
    protected final ScheduledExecutorService executorService;
    protected boolean running;

    public AbstractSubscriber(SubscriberConfiguration subscriberConfiguration, Destination destination, MessageListener messageListener, MessageContextFactory messageContextFactory, int pollingThreads) {
        this.subscriberConfiguration = subscriberConfiguration;
        this.destination = destination;
        this.messageListener = messageListener;
        this.messageContextFactory = messageContextFactory;
        if (pollingThreads < 1 || pollingThreads > 5) {
            throw new IllegalArgumentException("Polling threads must be between 1 to 5. Received: " + pollingThreads);
        }
        this.executorService = Executors.newScheduledThreadPool(pollingThreads);
    }

    public synchronized void start() {
        running = true;
        LOGGER.trace("Starting Subscriber");
        submitWork();
    }

    protected void submitWork() {
        if (running) {
            doSubmitWork();
        }
    }

    protected abstract void doSubmitWork();

    protected void handleError(Throwable e) {
        if (!AbstractCourierRestClient.isTimeout(e)) {
            messageListener.onError(e);
        }
    }

    protected void processMessages(List<AnypointMqMessage> messages) {
        if (running) { // TODO: Check this, because messages are already taken from the queue
            for (AnypointMqMessage message : messages) {
                AnypointMQMessageContext messageContext = messageContextFactory.createMessageContext(message, destination.getName());
                if (subscriberConfiguration.getAcknowledgementMode() == IMMEDIATE) {
                    destination.ack(messageContext.getMessage()).fireAndForget();
                }
                messageListener.onReceive(messageContext);
            }
        }
    }

    public synchronized void stop() {
        running = false;
        LOGGER.trace("Stopping Subscriber");
        stopExecutorService(executorService);
    }
}
