package com.mulesoft.extension.mq.internal;

import static com.mulesoft.mq.restclient.api.Destination.DEFAULT_LOCK_TTL;

import com.mulesoft.extension.mq.api.message.MessageContextFactory;
import com.mulesoft.extension.mq.internal.config.SubscriberConfiguration;
import com.mulesoft.extension.mq.internal.domain.MessageListener;
import com.mulesoft.mq.restclient.api.AnypointMqMessage;
import com.mulesoft.mq.restclient.api.CourierObserver;
import com.mulesoft.mq.restclient.api.Destination;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PollingSubscriber extends AbstractSubscriber {

  private static final Logger LOGGER = LoggerFactory.getLogger(PollingSubscriber.class);
  private static final int POLLING_THREADS = 2;
  private static final int BATCH_SIZE = 10;
  private final long pollingTime;

  public PollingSubscriber(SubscriberConfiguration subscriberConfiguration, Destination destination,
                           MessageListener messageListener, MessageContextFactory messageContextFactory) {
    super(subscriberConfiguration, destination, messageListener, messageContextFactory, POLLING_THREADS);
    this.pollingTime = subscriberConfiguration.getPollingTime() == null || subscriberConfiguration.getPollingTime() <= 0
      ? DEFAULT_POLLING_TIME
      : subscriberConfiguration.getPollingTime();
  }

  @Override
  protected void doSubmitWork() {
    executorService.scheduleAtFixedRate(() -> {
      try {
        subscribeForMessages();
      } catch (Throwable t) {
        LOGGER.error("Can not subscribe for messages.", t);
        throw t;
      }
    }, 0, pollingTime, TimeUnit.MILLISECONDS);
  }

  private void subscribeForMessages() {
    destination.receive(BATCH_SIZE,
                        Math.min(pollingTime / 2, 20),
                        Optional.ofNullable(subscriberConfiguration.getAcknowledgementTimeout()).orElse(DEFAULT_LOCK_TTL))
      .subscribe(new CourierObserver<List<AnypointMqMessage>>() {

        @Override
        public void onSuccess(List<AnypointMqMessage> messages) {
          processMessages(messages);
        }

        @Override
        public void onError(Throwable e) {
          handleError(e);
        }
      });
  }

}
