package com.mulesoft.extension.mq.internal.connection;

import static org.mule.runtime.api.connection.ConnectionValidationResult.failure;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;
import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;

import javax.inject.Inject;

public class AnypointMQConnectionProvider implements CachedConnectionProvider<AnypointMQConnection> {

    /**
     * The region URL where the Queue resides. This URL can be obtained and configured from the Anypoint Platform &gt; MQ console.
     * Copy/paste the region URL into this field."
     */
    @Parameter
    @Optional(defaultValue = "https://mq-us-east-1.anypoint.mulesoft.com/api/v1")
    @DisplayName("URL")
    @Placement(order = 1)
    @Example("https://mq-us-east-1.anypoint.mulesoft.com/api/v1")
    private String url;

    /**
     * In Anypoint Platform &gt; MQ &gt; Client Apps, click an app name (or create a new app) and
     * click Copy for the Client App ID field. Paste this value in the Studio Client App ID field
     */
    @Parameter
    @DisplayName("Client App ID")
    @Placement(order = 2)
    private String clientId;

    /**
     * In Anypoint Platform &gt; MQ &gt; Client Apps, click an app name (or create a new app) and
     * click Copy for the Client Secret field. Paste this value in the Studio Client Secret field.
     */
    @Parameter
    @Placement(order = 3)
    @DisplayName("Client Secret")
    private String clientSecret;

    @Parameter
    @Placement(tab = "TLS/SSL")
    @DisplayName("TLS Context")
    @Optional
    private TlsContextFactory tlsContextFactory;

    @Parameter
    @Placement(tab = "Proxy")
    @Optional
    @NullSafe
    private ProxyParameterGroup proxyConfig;

    @Parameter
    @Placement(tab = ADVANCED_TAB)
    @DisplayName("TCP Client Socket Properties")
    @Optional
    @NullSafe
    private TcpClientSocketParameterGroup tcpClientSocketProperties;

    @Inject
    protected HttpService httpService;

    @Inject
    protected SchedulerService schedulerService;

    @Override
    public AnypointMQConnection connect() throws ConnectionException {
        HttpClientConfiguration httpClientConfiguration = new HttpClientConfiguration.Builder()
                .setTlsContextFactory(tlsContextFactory)
                .setProxyConfig(proxyConfig.toProxyConfig())
                .setClientSocketProperties(tcpClientSocketProperties.toTcpClientSocketProperties())
                .setName("amqHttpClient")
                .build();
        HttpClient httpClient = httpService.getClientFactory().create(httpClientConfiguration);
        return new AnypointMQConnection(httpClient, url, clientId, clientSecret, schedulerService.ioScheduler());
    }

    @Override
    public void disconnect(AnypointMQConnection connection) {
        connection.disconnect();
    }

    @Override
    public ConnectionValidationResult validate(AnypointMQConnection connection) {
        try {
            // TODO MQ-540 - Add validation
            connection.validate();
            return success();
        } catch (RuntimeException e) {
            return failure("An error occurred while validating the connection.", e);
        }
    }

    public void setHttpService(HttpService httpService) {
        this.httpService = httpService;
    }

    public void setSchedulerService(SchedulerService schedulerService) {
        this.schedulerService = schedulerService;
    }
}
