package com.mulesoft.extension.mq.internal.operation;

import com.mulesoft.extension.mq.api.attributes.AnypointMqMessagePublishAttributes;
import com.mulesoft.extension.mq.internal.error.AnypointMQErrorTypeProvider;
import com.mulesoft.extension.mq.api.message.AnypointMQMessageContext;
import com.mulesoft.extension.mq.internal.config.AnypointMQConfiguration;
import com.mulesoft.extension.mq.internal.config.ConsumerAckMode;
import com.mulesoft.extension.mq.internal.connection.AnypointMQConnection;
import com.mulesoft.extension.mq.internal.service.AnypointMQService;
import com.mulesoft.extension.mq.internal.service.AnypointMQServiceImpl;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.runtime.operation.Result;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Map;

import static com.mulesoft.mq.restclient.api.AnypointMqMessage.Properties.AMQ_MESSAGE_CONTENT_TYPE;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;

/**
 * Operations for the Anypoint MQ Connector
 *
 * @since 2.0.0
 */
@Throws(AnypointMQErrorTypeProvider.class)
public class AnypointMQOperations {

    /**
     * @param config          The Configuration of the connector.
     * @param connection      The connection to Anypoint MQ.
     * @param destination     Queue or Exchange name from where to fetch a Message
     * @param messageId       ID of the message to publish
     * @param sendContentType Indicates whether the content type of the Mule Message should be
     *                        attached or not
     * @param properties      Additional properties to be sent within the message
     * @param body         Body of the message
     * @return The same payload that the one consumed for the publish operations, but adds
     * attributes with metadata of the published message
     */
    @org.mule.runtime.extension.api.annotation.param.MediaType(value = "*/*", strict=false)
    public Result<InputStream, AnypointMqMessagePublishAttributes> publish(@Config AnypointMQConfiguration config,
                                                                           @Connection AnypointMQConnection connection,
                                                                           String destination,
                                                                           @Content(primary = true) @Optional(defaultValue = "#[payload]") TypedValue<InputStream> body,
                                                                           @Placement(tab = ADVANCED_TAB) @Optional String messageId,
                                                                           @Placement(tab = ADVANCED_TAB) @Optional(defaultValue = "true") boolean sendContentType,
                                                                           @Optional Map<String, String> properties) {
        MediaType mediaType = body.getDataType().getMediaType();
        byte[] messageContent = getContent(body.getValue());
        return Result.<InputStream, AnypointMqMessagePublishAttributes>builder()
                .output(new ByteArrayInputStream(messageContent))
                .mediaType(mediaType)
                .attributes(new AnypointMqMessagePublishAttributes(getService(config, connection).publish(destination,
                        messageContent,
                        sendContentType,
                        mediaType.toString(),
                        mediaType.getCharset(),
                        messageId,
                        properties)))
                .build();
    }

    /**
     * @param config                 The Configuration of the connector.
     * @param connection             The connection to Anypoint MQ.
     * @param destination            Queue or Exchange name from where to fetch a Message
     * @param acknowledgementMode    Acknowledgement mode to use for the messages retrieved from
     *                               this subscriber. Can be only used 'MANUAL' or 'NONE'.
     * @param pollingTime            How much time (milliseconds) to be waited if the requested
     *                               messages are not ready to be consumed.
     * @param acknowledgementTimeout Duration that a message is held by a broker waiting for an
     *                               Acknowledgement or Not Acknowledgement. After that duration
     *                               expires, the message is again available to any subscriber.
     * @return A {@code byte[]} with the body of the message and {@link AnypointMQMessageContext} with
     * the metadata of it.
     */
    @org.mule.runtime.extension.api.annotation.param.MediaType(value = "*/*", strict=false)
    public Result<InputStream, AnypointMQMessageContext> consume(@Config AnypointMQConfiguration config,
                                                                 @Connection AnypointMQConnection connection,
                                                                 String destination,
                                                                 @Optional(defaultValue = "MANUAL") ConsumerAckMode acknowledgementMode,
                                                                 @Optional(defaultValue = "10000") Long pollingTime,
                                                                 @Optional(defaultValue = "0") Long acknowledgementTimeout) {
        Result.Builder<InputStream, AnypointMQMessageContext> resultBuilder = Result.builder();
        AnypointMQMessageContext messageContext = getService(config, connection).consume(destination, acknowledgementMode, pollingTime, acknowledgementTimeout);
        if (messageContext != null) {
            resultBuilder.output(new ByteArrayInputStream(messageContext.getMessage().getBody()))
                    .attributes(messageContext);
            Map<String, String> properties = messageContext.getMessage().getProperties();
            if (properties.containsKey(AMQ_MESSAGE_CONTENT_TYPE)) {
                resultBuilder.mediaType(MediaType.parse(properties.get(AMQ_MESSAGE_CONTENT_TYPE)));
            }
        }

        return resultBuilder.build();
    }

    /**
     * Executes an Acknowledgement over a given {@link AnypointMQMessageContext} indicating that the
     * message has been consumed correctly and deletes the message from {@code In Flight} status.
     *
     * @param config         The Configuration of the connector.
     * @param connection     The connection to Anypoint MQ.
     * @param messageContext {@link AnypointMQMessageContext} that represents the received message
     */
    public void ack(@Config AnypointMQConfiguration config,
                    @Connection AnypointMQConnection connection,
                    @Expression(value = REQUIRED) AnypointMQMessageContext messageContext) {
        getService(config, connection).ack(messageContext);
    }

    /**
     * Executes an Not Acknowledgement over a given {@link AnypointMQMessageContext} and change the
     * status of the message from {@code In Flight} to {@code In Queue} to be consumed again for a
     * subscriber
     *
     * @param config         The Configuration of the connector.
     * @param connection     The connection to Anypoint MQ.
     * @param messageContext {@link AnypointMQMessageContext} that represents the received message
     */
    public void nack(@Config AnypointMQConfiguration config,
                     @Connection AnypointMQConnection connection,
                     @Expression(value = REQUIRED) AnypointMQMessageContext messageContext) {
        getService(config, connection).nack(messageContext);
    }

    private AnypointMQService getService(AnypointMQConfiguration config, AnypointMQConnection connection) {
        return new AnypointMQServiceImpl(config, connection);
    }

    private byte[] getContent(InputStream inputStream) {
        if (inputStream != null) {
            return org.mule.runtime.core.api.util.IOUtils.toByteArray(inputStream);
        } else {
            return new byte[0];
        }
    }
}
