package com.mulesoft.extension.mq.internal.server;

import static com.mulesoft.extension.mq.internal.config.SubscriberAckMode.AUTO;
import static com.mulesoft.extension.mq.internal.config.SubscriberAckMode.IMMEDIATE;
import static com.mulesoft.extension.mq.internal.config.SubscriberConfiguration.DEFAULT_MAX_REDELIVERY;
import static com.mulesoft.mq.restclient.api.AnypointMqMessage.Properties.AMQ_MESSAGE_CONTENT_TYPE;
import static com.mulesoft.mq.restclient.api.Destination.DEFAULT_LOCK_TTL;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import com.mulesoft.extension.mq.api.message.AnypointMQMessageContext;
import com.mulesoft.extension.mq.internal.AbstractSubscriber;
import com.mulesoft.extension.mq.internal.PollingSubscriber;
import com.mulesoft.extension.mq.internal.PrefetchSubscriber;
import com.mulesoft.extension.mq.internal.config.AnypointMQConfiguration;
import com.mulesoft.extension.mq.internal.config.MQPrefetchConfiguration;
import com.mulesoft.extension.mq.internal.config.SubscriberAckMode;
import com.mulesoft.extension.mq.internal.config.SubscriberConfiguration;
import com.mulesoft.extension.mq.internal.connection.AnypointMQConnection;
import com.mulesoft.extension.mq.internal.domain.MessageListener;
import com.mulesoft.mq.restclient.api.Destination;
import com.mulesoft.mq.restclient.api.exception.ResourceNotFoundException;
import com.mulesoft.mq.restclient.impl.PrefetchedDestination;
import com.mulesoft.mq.restclient.internal.DefaultMessagePreserver;
import com.mulesoft.mq.restclient.internal.ScheduledPrefetcher;
import com.mulesoft.mq.restclient.internal.TimeSupplier;

import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AnypointMQServer {

    private static final Logger LOGGER = LoggerFactory.getLogger(AnypointMQServer.class);
    public static final String MESSAGE_CONTEXT_VAR = "MESSAGE_CONTEXT";
    public static final String DESTINATION_VAR = "DESTINATION";
    public static final String ACKNOWLEDGEMENT_MODE_VAR = "ACKNOWLEDGEMENT_MODE";

    private final AnypointMQConnection connection;
    private final SourceCallback<byte[], AnypointMQMessageContext> callback;
    private AbstractSubscriber subscriber;
    private ScheduledPrefetcher prefetcher;

    public AnypointMQServer(String destination, AnypointMQConfiguration config, AnypointMQConnection connection, SourceCallback<byte[], AnypointMQMessageContext> callback) {
        this.connection = connection;
        this.callback = callback;
        Destination courierDestination = connection.getDestination(destination);

        MQPrefetchConfiguration prefetch = config.getPrefetch();
        if (prefetch != null && prefetch.getFetchSize() > 0) {
            prefetcher = new ScheduledPrefetcher(courierDestination,
                    prefetch.getFetchSize(),
                    prefetch.getFetchTimeout(),
                    Optional.ofNullable(config.getAcknowledgementTimeout()).orElse(DEFAULT_LOCK_TTL),
                    prefetch.getFrequency(),
                    new DefaultMessagePreserver(courierDestination, new TimeSupplier()));
            courierDestination = new PrefetchedDestination(courierDestination, prefetcher);
            subscriber = new PrefetchSubscriber(config, courierDestination, new AnypointMQServer.ExtensionMessageListener(config, courierDestination, this.callback), connection.getMessageContextFactory());
        } else {
            subscriber = new PollingSubscriber(config, courierDestination, new AnypointMQServer.ExtensionMessageListener(config, courierDestination, this.callback), connection.getMessageContextFactory());
        }
    }

    public void start(){
        if (subscriber != null) {
            subscriber.start();
        }
        
        if (prefetcher != null){
            prefetcher.start();
        }
    }

    public void stop() {
        connection.disconnect();
        if (prefetcher != null) {
            prefetcher.stop();
        }

        if (subscriber != null) {
            subscriber.stop();
        }
    }

    private class ExtensionMessageListener implements MessageListener {
        private final SubscriberAckMode acknowledgementMode;
        private final Destination destination;
        private final SourceCallback<byte[], AnypointMQMessageContext> callback;
        private final int maxRedelivery;

        ExtensionMessageListener(SubscriberConfiguration subscriberConfiguration, Destination destination, SourceCallback<byte[], AnypointMQMessageContext> callback) {
            this.acknowledgementMode = Optional.ofNullable(subscriberConfiguration.getAcknowledgementMode()).orElse(AUTO);
            this.maxRedelivery = Optional.ofNullable(subscriberConfiguration.getMaxRedelivery()).orElse(DEFAULT_MAX_REDELIVERY);
            this.destination = destination;
            this.callback = callback;
        }

        @Override
        public void onReceive(AnypointMQMessageContext messageContext) {
            if (redeliveryExhausted(messageContext, maxRedelivery)) {
                LOGGER.trace("Listener Skipped Message '{}' - Max redelivery reached with '{}' attempts",
                             messageContext.getMessage().getId(), messageContext.getMessage().getDeliveryCount());
                return;
            }

            if (acknowledgementMode == IMMEDIATE) {
                destination.ack(messageContext.getMessage()).fireAndForget();
            }

            LOGGER.trace("Message received - {}", messageContext.getMessage().getId());
            handleMessage(messageContext);
        }

        private void handleMessage(AnypointMQMessageContext messageContext) {
            Map<String, String> properties = messageContext.getMessage().getProperties();

            Result.Builder<byte[], AnypointMQMessageContext> resultbuilder = Result.<byte[], AnypointMQMessageContext>builder()
                    .output(messageContext.getMessage().getBody())
                    .attributes(messageContext);

            if (properties.containsKey(AMQ_MESSAGE_CONTENT_TYPE)) {
                resultbuilder.mediaType(MediaType.parse(properties.get(AMQ_MESSAGE_CONTENT_TYPE)));
            }

            SourceCallbackContext callbackContext = callback.createContext();
            callbackContext.addVariable(ACKNOWLEDGEMENT_MODE_VAR, acknowledgementMode);
            callbackContext.addVariable(DESTINATION_VAR, destination);
            callbackContext.addVariable(MESSAGE_CONTEXT_VAR, messageContext);
            callback.handle(resultbuilder.build(), callbackContext);
            
            LOGGER.trace("Message dispatched - {}", messageContext.getMessage().getId());
        }

        @Override
        public void onError(Throwable throwable) {
            if (throwable instanceof ResourceNotFoundException) {
                LOGGER.error("Connection failed - Destination not found: {}", destination);
                callback.onConnectionException(new ConnectionException(throwable));
            } else {
                LOGGER.error("Can not process received message.", throwable);
            }
        }

        private boolean redeliveryExhausted(AnypointMQMessageContext messageContext, int maxRedelivery) {
            boolean exhausted = false;
            if (maxRedelivery >= 0) {
                int redeliveries = messageContext.getMessage().getDeliveryCount() - 1;
                if (redeliveries > maxRedelivery) {
                    exhausted = true;
                }
            }
            return exhausted;
        }

    }
}
