package com.mulesoft.extension.mq.internal.service;

import static com.mulesoft.mq.restclient.api.AnypointMqMessage.Properties.AMQ_MESSAGE_CONTENT_TYPE;

import com.mulesoft.extension.mq.api.message.AnypointMQMessageContext;
import com.mulesoft.extension.mq.internal.config.AnypointMQConfiguration;
import com.mulesoft.extension.mq.internal.config.ConsumerAckMode;
import com.mulesoft.extension.mq.internal.connection.AnypointMQConnection;
import com.mulesoft.mq.restclient.api.AnypointMqMessage;
import com.mulesoft.mq.restclient.api.AnypointMqMessageBuilder;
import com.mulesoft.mq.restclient.api.Destination;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import static com.mulesoft.extension.mq.internal.config.ConsumerAckMode.IMMEDIATE;

public class AnypointMQServiceImpl implements AnypointMQService {

    private final AnypointMQConfiguration config;
    private final AnypointMQConnection connection;

    public AnypointMQServiceImpl(AnypointMQConfiguration config, AnypointMQConnection connection) {
        this.config = config;
        this.connection = connection;
    }

    @Override
    public void ack(AnypointMQMessageContext messageContext) {
        connection.getDestination(messageContext.getDestination()).ack(messageContext.getMessage()).fireAndForget();
    }

    @Override
    public void nack(AnypointMQMessageContext messageContext) {
        connection.getDestination(messageContext.getDestination()).nack(messageContext.getMessage()).fireAndForget();
    }

    @Override
    public AnypointMQMessageContext consume(String destination, ConsumerAckMode acknowledgementMode, Long pollingTime, Long acknowledgementTimeout) {
        Destination courierDestination = connection.getDestination(destination);

        // TODO: MQ-365 Receive poolingTime parameter and use it to pool the destination
        List<AnypointMqMessage> messages = courierDestination.receive(1, pollingTime, acknowledgementTimeout).getValue();
        AnypointMQMessageContext messageContext = null;
        if (messages != null && messages.size() == 1) {
            messageContext = connection.getMessageContextFactory().createMessageContext(messages.get(0), destination);
            if (acknowledgementMode == IMMEDIATE) {
                ack(messageContext);
            }
        }
        return messageContext;
    }

    @Override
    public String publish(String destination, byte[] content, boolean sendContentType, String mediaType, Optional<Charset> charset, String messageId, Map<String, String> properties) {
        Destination courierDestination = connection.getDestination(destination);
        AnypointMqMessageBuilder messageBuilder = courierDestination.newMessageBuilder();

        String id = Optional.<Object>ofNullable(messageId).orElseGet(UUID::randomUUID).toString();
        messageBuilder.withMessageId(id);
        messageBuilder.withBody(content);

        if (properties != null) {
            for (Map.Entry<String, String> entry : properties.entrySet()) {
                messageBuilder.addProperty(entry.getKey(), entry.getValue());
            }
        }

        if (sendContentType) {
            messageBuilder.addProperty(AMQ_MESSAGE_CONTENT_TYPE, mediaType.toString());
            charset.map(Object::toString)
                    .ifPresent(value -> messageBuilder.addProperty("MULE_ENCODING", value));
        }

        // TODO switching to a blocking call in order to throw the correct exception, fix it and make it async and the publisher an async mp
        courierDestination.send(messageBuilder.build()).getValue();
        return id;
    }


}
