package org.mule.extension.aws.commons.internal.connection.provider;


import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.client.builder.AwsAsyncClientBuilder;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.metrics.AwsSdkMetrics;
import org.mule.connectors.commons.template.connection.ConnectorConnectionProvider;
import org.mule.extension.aws.commons.internal.connection.AWSConnection;
import org.mule.extension.aws.commons.internal.connection.provider.parameter.CommonParameters;
import org.mule.extension.aws.commons.internal.connection.provider.parameter.ProxyParameterGroup;
import org.mule.extension.aws.commons.internal.connection.provider.parameter.RegionEndpoint;
import org.mule.extension.aws.commons.internal.exception.AWSConnectionException;
import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Placement;

import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Predicate;

public abstract class AWSConnectionProvider<AWS_CLIENT,
        AWS_ASYNC_CLIENT,
        AWS_CLIENT_BUILDER extends AwsClientBuilder<AWS_CLIENT_BUILDER, AWS_CLIENT>,
        AWS_ASYNC_CLIENT_BUILDER extends AwsAsyncClientBuilder<AWS_ASYNC_CLIENT_BUILDER, AWS_ASYNC_CLIENT>,
        CONNECTION extends AWSConnection<AWS_CLIENT, AWS_ASYNC_CLIENT>> extends ConnectorConnectionProvider implements CachedConnectionProvider<CONNECTION> {

    private final AWS_CLIENT_BUILDER clientBuilder;
    private final AWS_ASYNC_CLIENT_BUILDER asyncClientBuilder;
    private final BiFunction<AWS_CLIENT, AWS_ASYNC_CLIENT, CONNECTION> connectionConstructor;

    @ParameterGroup(name = "Connection")
    @Placement(order = 1)
    private CommonParameters commonParameters;

    @ParameterGroup(name = "Proxy")
    @Placement(order = 2)
    private ProxyParameterGroup proxyParameterGroup;

    public AWSConnectionProvider(BiFunction<AWS_CLIENT, AWS_ASYNC_CLIENT, CONNECTION> connectionConstructor, AWS_CLIENT_BUILDER clientBuilder, AWS_ASYNC_CLIENT_BUILDER asyncClientBuilder) {
        this.connectionConstructor = connectionConstructor;
        this.clientBuilder = clientBuilder;
        this.asyncClientBuilder = asyncClientBuilder;
    }

    @Override
    public CONNECTION connect() throws ConnectionException {
        try {
            ClientConfiguration clientConfiguration = new ClientConfiguration();
            if (proxyParameterGroup != null) {
                Optional.ofNullable(proxyParameterGroup.getProxyDomain()).ifPresent(clientConfiguration::setProxyDomain);
                Optional.ofNullable(proxyParameterGroup.getProxyUsername()).ifPresent(clientConfiguration::setProxyUsername);
                Optional.ofNullable(proxyParameterGroup.getProxyHost()).ifPresent(clientConfiguration::setProxyHost);
                Optional.ofNullable(proxyParameterGroup.getProxyPassword()).ifPresent(clientConfiguration::setProxyPassword);
                Optional.ofNullable(proxyParameterGroup.getProxyPort()).ifPresent(clientConfiguration::setProxyPort);
                Optional.ofNullable(proxyParameterGroup.getProxyWorkstation()).ifPresent(clientConfiguration::setProxyWorkstation);
            }
            clientConfiguration.setSocketTimeout(commonParameters.getSocketTimeout());
            clientConfiguration.setConnectionTimeout(commonParameters.getConnectionTimeout());
            configureRegionProperty(clientBuilder, asyncClientBuilder);
            appendClientConfigurationProperties(clientConfiguration);
            clientBuilder.withClientConfiguration(clientConfiguration);
            asyncClientBuilder.withClientConfiguration(clientConfiguration);
            if (!commonParameters.isTryDefaultAWSCredentialsProviderChain() &&
                    (!Optional.ofNullable(commonParameters.getAccessKey()).filter(Predicate.isEqual("").negate()).isPresent() || !Optional.ofNullable(commonParameters.getSecretKey()).filter(Predicate.isEqual("").negate()).isPresent())) {
                throw new AWSConnectionException("Access Key or Secret Key is blank");
            }
            clientBuilder.withCredentials(getAWSCredentialsProvider(commonParameters));
            asyncClientBuilder.withCredentials(getAWSCredentialsProvider(commonParameters));
            appendBuilderProperties(clientBuilder, asyncClientBuilder);

            // Unregistering the Metric Admin MBean to avoid a memory leak.
            AwsSdkMetrics.unregisterMetricAdminMBean();
            CONNECTION connection = connectionConstructor.apply(clientBuilder.build(), asyncClientBuilder.build());
            onConnect(connection);
            return connection;

        } catch (AWSConnectionException e) {
            throw new ConnectionException(e.getMessage(), e);
        }
    }

    protected void onConnect(CONNECTION connection) {
        // Implementations that want to do something after the connection has been created should overwrite this.
    }

    protected void appendBuilderProperties(AWS_CLIENT_BUILDER clientBuilder, AWS_ASYNC_CLIENT_BUILDER asyncClientBuilder) {
        // Implementations that want to append properties to the builder should overwrite this.
    }

    protected void configureRegionProperty(AWS_CLIENT_BUILDER clientBuilder, AWS_ASYNC_CLIENT_BUILDER asyncClientBuilder){
        // Override this method to set Endpoint Configuration (S3 storageURL)
        String region = RegionEndpoint.valueOf(commonParameters.getRegion()).toString();
        clientBuilder.setRegion(region);
        asyncClientBuilder.setRegion(region);
    }

    protected abstract AWSCredentialsProvider getAWSCredentialsProvider(CommonParameters commonParameters);

    protected void appendClientConfigurationProperties(ClientConfiguration configuration) {
        // Implementations that want to append properties to the client configuration should overwrite this.
    }

    @Override
    public void disconnect(CONNECTION connection) {
        super.disconnect(connection);
    }

    @Override
    public ConnectionValidationResult validate(CONNECTION connection) {
        return super.validate(connection);
    }

    public CommonParameters getCommonParameters() {
        return commonParameters;
    }

    public void setCommonParameters(CommonParameters commonParameters) {
        this.commonParameters = commonParameters;
    }

    public void setProxyParameterGroup(ProxyParameterGroup proxyParameterGroup) {
        this.proxyParameterGroup = proxyParameterGroup;
    }
}
