/**
 * (c) 2003-2019 MuleSoft, Inc. The software in this package is
 * published under the terms of the Commercial Free Software license V.1, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.extension.rds.internal.operation;

import com.amazonaws.services.rds.model.DescribeDBInstancesRequest;
import org.mule.extension.rds.api.attributes.RequestIDAttribute;
import org.mule.extension.rds.api.model.DBInstance;
import org.mule.extension.rds.api.model.Filter;
import org.mule.extension.rds.api.model.Tag;
import org.mule.extension.rds.internal.config.RDSConfiguration;
import org.mule.extension.rds.internal.connection.RDSConnection;
import org.mule.extension.rds.internal.error.RDSErrorTypeProvider;
import org.mule.extension.rds.internal.operation.group.MonitoringParameterGroup;
import org.mule.extension.rds.internal.operation.group.dbinstance.CreationDBInstanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.dbinstance.ModificationDBInstanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.security.CreationSecurityParameterGroup;
import org.mule.extension.rds.internal.operation.group.security.ModificationSecurityParameterGroup;
import org.mule.extension.rds.internal.operation.group.security.SecurityParameterGroup;
import org.mule.extension.rds.internal.operation.group.storage.CreationStorageAndMaintenanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.storage.ModificationStorageAndMaintenanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.storage.StorageParameterGroup;
import org.mule.extension.rds.internal.operation.paging.DescribeDBInstancesPagingProvider;
import org.mule.extension.rds.internal.service.DBInstanceService;
import org.mule.extension.rds.internal.service.DBInstanceServiceImpl;
import org.mule.extension.rds.internal.util.RDSModelFactory;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

@Throws(RDSErrorTypeProvider.class)
public class DBInstanceOperations extends RDSOperations<DBInstanceService> {

    public DBInstanceOperations() {
        super(DBInstanceServiceImpl::new);
    }

    /**
     * <p>
     * Creates a new DB instance.
     * </p>
     * <p><a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>API Reference</a></p>
     *
     * @param config                                Configuration for RDS connector.
     * @param client                                Amazon RDS Client connection instance.
     * @param storageAndMaintenanceParameterGroup   Group of parameters for setting up storage and maintenance for DB instance.
     * @param securityParameterGroup                Group of parameters for setting up subnet group for DB instance, name of the IAM role to be used when making API calls to the Directory Service,
     *                                              flag to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts, license model information for the restored DB instance,
     *                                              accessibility options for the DB instance and TDE encryption credentials.
     * @param monitoringParameterGroup              Group of parameters for setting up the monitoring interval (in seconds) and ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs
     * @param dbInstanceParameterGroup              Group of parameters for setting up the port number on which the database accepts connections and new DB instance identifier and others DB instance parameters.
     * @return DBInstance
     */
    public DBInstance createDbInstance(@Config RDSConfiguration config,
                                       @Connection RDSConnection client,
                                       @ParameterGroup(name = "Storage and Maintenance") CreationStorageAndMaintenanceParameterGroup storageAndMaintenanceParameterGroup,
                                       @ParameterGroup(name = "Security") CreationSecurityParameterGroup securityParameterGroup,
                                       @ParameterGroup(name = "Monitoring") MonitoringParameterGroup monitoringParameterGroup,
                                       @ParameterGroup(name = "DB Instance") CreationDBInstanceParameterGroup dbInstanceParameterGroup) {
        return newExecutionBuilder(config, client).execute(DBInstanceService::createDbInstance)
                .withParam(storageAndMaintenanceParameterGroup)
                .withParam(securityParameterGroup)
                .withParam(monitoringParameterGroup)
                .withParam(dbInstanceParameterGroup);
    }

    /**
     * Modifies settings for a DB instance. You can change one or more database configuration parameters by specifying
     * these parameters and the new values in the request.
     * <p><a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBInstance.html>API Reference</a></p>
     *
     * @param config                                Configuration for RDS connector.
     * @param client                                Amazon RDS Client connection instance.
     * @param storageAndMaintenanceParameterGroup   Group of parameters for setting up storage and maintenance for DB instance.
     * @param securityParameterGroup                Group of parameters for setting up subnet group for DB instance, name of the IAM role to be used when making API calls to the Directory Service,
     *                                              flag to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts, license model information for the restored DB instance,
     *                                              accessibility options for the DB instance and TDE encryption credentials.
     * @param monitoringParameterGroup              Group of parameters for setting up the monitoring interval (in seconds) and ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs
     * @param dbInstanceParameterGroup              Group of parameters for setting up the port number on which the database accepts connections and new DB instance identifier and others DB instance parameters.
     * @return DBInstance
     */
    public DBInstance modifyDbInstance(@Config RDSConfiguration config,
                                       @Connection RDSConnection client,
                                       @ParameterGroup(name = "Storage and Maintenance") ModificationStorageAndMaintenanceParameterGroup storageAndMaintenanceParameterGroup,
                                       @ParameterGroup(name = "Security") ModificationSecurityParameterGroup securityParameterGroup,
                                       @ParameterGroup(name = "Monitoring") MonitoringParameterGroup monitoringParameterGroup,
                                       @ParameterGroup(name = "DB Instance") ModificationDBInstanceParameterGroup dbInstanceParameterGroup) {
        return newExecutionBuilder(config, client).execute(DBInstanceService::modifyDbInstance)
                .withParam(storageAndMaintenanceParameterGroup)
                .withParam(securityParameterGroup)
                .withParam(monitoringParameterGroup)
                .withParam(dbInstanceParameterGroup);
    }

    /**
     * <p>The DeleteDBInstance action deletes a previously provisioned DB instance.</p>
     * <p> <a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBInstance.html>API Reference</a></p>
     *
     * @param config                    Configuration for RDS connector.
     * @param client                    Amazon RDS Client connection instance.
     * @param dbInstanceIdentifier      The DB instance identifier for the DB instance to be deleted. This parameter isn't case-sensitive.
     * @param finalDbSnapshotIdentifier The DBSnapshotIdentifier of the new DBSnapshot created when SkipFinalSnapshot is set to false.
     * @param skipFinalSnapshot         Determines whether a final DB snapshot is created before the DB instance is deleted.
     * @return DBInstance
     */
    public DBInstance deleteDbInstance(@Config RDSConfiguration config,
                                       @Connection RDSConnection client,
                                       @DisplayName("DB Instance Identifier") String dbInstanceIdentifier,
                                       @DisplayName("Final DB Snapshot Identifier") @Optional String finalDbSnapshotIdentifier,
                                       @DisplayName("Skip Final Snapshot") @Optional boolean skipFinalSnapshot) {
        return newExecutionBuilder(config, client).execute(DBInstanceService::deleteDBInstance)
                .withParam(dbInstanceIdentifier)
                .withParam(finalDbSnapshotIdentifier)
                .withParam(skipFinalSnapshot);
    }

    /**
     * <p>
     * Starts a DB instance that was stopped using the AWS console, the stop-db-instance AWS CLI command, or the StopDBInstance action
     * </p>
     * <p> <a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StartDBInstance.html>API Reference</a></p>
     *
     * @param config               Configuration for RDS connector.
     * @param client               Amazon RDS Client connection instance.
     * @param dbInstanceIdentifier The user-supplied instance identifier.
     * @return DBInstance
     */
    public DBInstance startDbInstance(@Config RDSConfiguration config,
                                      @Connection RDSConnection client,
                                      @DisplayName("DB Instance Identifier") String dbInstanceIdentifier) {
        return newExecutionBuilder(config, client).execute(DBInstanceService::startDBInstance)
                .withParam(dbInstanceIdentifier);
    }

    /**
     * <p>
     * Stops a DB instance.
     * </p>
     * <p> <a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_StopDBInstance.html>API Reference</a></p>
     *
     * @param config               Configuration for RDS connector.
     * @param client               Amazon RDS Client connection instance.
     * @param dbInstanceIdentifier The user-supplied instance identifier.
     * @param dbSnapshotIdentifier The user-supplied instance identifier of the DB Snapshot created immediately before the DB instance is stopped.
     * @return DBInstance
     */
    public DBInstance stopDbInstance(@Config RDSConfiguration config,
                                     @Connection RDSConnection client,
                                     @DisplayName("DB Instance Identifier") String dbInstanceIdentifier,
                                     @DisplayName("DB Snapshot Identifier") @Optional String dbSnapshotIdentifier) {
        return newExecutionBuilder(config, client).execute(DBInstanceService::stopDBInstance)
                .withParam(dbInstanceIdentifier)
                .withParam(dbSnapshotIdentifier);
    }

    /**
     * <p>
     * Rebooting a DB instance restarts the database engine service.
     * </p>
     * <p> <a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RebootDBInstance.html>API Reference</a></p>
     *
     * @param config               Configuration for RDS connector.
     * @param client               Amazon RDS Client connection instance.
     * @param dbInstanceIdentifier The DB instance identifier.
     * @param forceFailover        When true, the reboot will be conducted through a MultiAZ failover.
     * @return DBInstance
     */
    public DBInstance rebootDbInstance(@Config RDSConfiguration config,
                                       @Connection RDSConnection client,
                                       @DisplayName("DB Instance Identifier") String dbInstanceIdentifier,
                                       @DisplayName("Force Failover") @Optional boolean forceFailover) {
        return newExecutionBuilder(config, client).execute(DBInstanceService::rebootDBInstance)
                .withParam(dbInstanceIdentifier)
                .withParam(forceFailover);
    }

    /**
     * <p>
     * Returns information about provisioned RDS instances. This API supports pagination.
     * </p>
     * <p> <a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html>API Reference</a></p>
     *
     * @param dbInstanceIdentifier The user-supplied instance identifier.
     * @param filters              A filter that specifies one or more DB instances to describe.
     * @return DescribeDBInstancesResult
     */
    public PagingProvider<RDSConnection, Result<DBInstance, RequestIDAttribute>> describeDbInstances(@DisplayName("DB Instance Identifier") @Optional String dbInstanceIdentifier,
                                                                                                     @Optional Collection<Filter> filters) {
        return new DescribeDBInstancesPagingProvider(new DescribeDBInstancesRequest()
                .withDBInstanceIdentifier(dbInstanceIdentifier)
                .withFilters(RDSModelFactory.unWrapFilterList(filters)));
    }

    /**
     * <p>
     * Creates a new DB instance from a DB snapshot.
     * </p>
     * <p> <a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceFromDBSnapshot.html>API Reference</a></p>
     *
     * @param config                    Configuration for RDS connector.
     * @param connection                Amazon RDS Client connection instance.
     * @param storageParameterGroup     Group of parameters for setting up auto minor version upgrade, storage type for read replica and a flag indicating whether to copy tags to read replica snapshots
     * @param securityParameterGroup    Group of parameters for setting up subnet group for DB instance, name of the IAM role to be used when making API calls to the Directory Service,
     *                                  flag to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts, license model information for the restored DB instance,
     *                                  accessibility options for the DB instance and TDE encryption credentials.
     * @param availabilityZone          The EC2 Availability Zone that the database instance will be created in.
     * @param dbInstanceClass           The compute and memory capacity of the Amazon RDS DB instance.
     * @param dbInstanceIdentifier      Name of the DB instance to create from the DB snapshot. This parameter isn't case-sensitive.
     * @param dbSnapshotIdentifier      The identifier for the DB snapshot to restore from.
     * @param dbName                    The database name for the restored DB instance.
     * @param domain                    Specify the Active Directory Domain to restore the instance in.
     * @param engine                    The database engine to use for the new instance.
     * @param iops                      Specifies the amount of provisioned IOPS for the DB instance, expressed in I/O operations per second.
     * @param multiAZ                   Specifies if the DB instance is a Multi-AZ deployment.
     * @param optionGroupName           The name of the option group to be used for the restored DB instance.
     * @param port                      The port number on which the database accepts connections.
     * @param tags                      A list of tags.
     * @return DBInstance
     */
    public DBInstance restoreDbInstanceFromDbSnapshot(@Config RDSConfiguration config,
                                                      @Connection RDSConnection connection,
                                                      @ParameterGroup(name = STORAGE_AND_MAINTENANCE) StorageParameterGroup storageParameterGroup,
                                                      @ParameterGroup(name = SECURITY) SecurityParameterGroup securityParameterGroup,
                                                      @DisplayName("Availability Zone") @Placement(order = 1) @Optional String availabilityZone,
                                                      @DisplayName("DB Instance Class") @Placement(order = 2) @Optional String dbInstanceClass,
                                                      @DisplayName("DB Instance Identifier") @Placement(order = 3) String dbInstanceIdentifier,
                                                      @DisplayName("DB Snapshot Identifier") @Placement(order = 4) String dbSnapshotIdentifier,
                                                      @DisplayName("DB Name") @Placement(order = 5) @Optional String dbName,
                                                      @Placement(order = 6) @Optional String domain,
                                                      @Placement(order = 7) @Optional String engine,
                                                      @DisplayName("IOPS") @Placement(order = 8) @Optional Integer iops,
                                                      @DisplayName("Multi AZ") @Placement(order = 9) @Optional(defaultValue = "false") boolean multiAZ,
                                                      @DisplayName("Option Group Name") @Placement(order = 10) @Optional String optionGroupName,
                                                      @Placement(order = 11) @Optional Integer port,
                                                      @Placement(order = 12) @Optional List<Tag> tags) {
        // FIXME: POST GA: Adapt to Atlantic Commons.
        return new DBInstanceServiceImpl(config, connection).restoreDbInstanceFromDbSnapshot(storageParameterGroup.isAutoMinorVersionUpgrade(),
                availabilityZone,
                storageParameterGroup.isCopyTagsToSnapshot(),
                dbInstanceClass,
                dbInstanceIdentifier,
                dbSnapshotIdentifier,
                dbName,
                securityParameterGroup.getDbSubnetGroupName(),
                domain,
                securityParameterGroup.getDomainIAMRoleName(),
                securityParameterGroup.isEnableIAMDatabaseAuthentication(),
                engine,
                iops,
                securityParameterGroup.getLicenseModel(),
                multiAZ,
                optionGroupName, port,
                securityParameterGroup.isPubliclyAccessible(),
                storageParameterGroup.getStorageType(),
                tags,
                securityParameterGroup.getTdeCredentialArn(),
                securityParameterGroup.getTdeCredentialPassword());
    }

    /**
     * Restores a DB instance to an arbitrary point in time. You can restore to any point in time before the time
     * identified by the LatestRestorableTime property. You can restore to a point up to the number of days specified
     * by the BackupRetentionPeriod property.
     * <p><a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_RestoreDBInstanceToPointInTime.html>API Reference</a></p>
     *
     * @param config                     Configuration for RDS connector.
     * @param connection                 Amazon RDS Client connection instance.
     * @param storageParameterGroup      Group of parameters for setting up auto minor version upgrade, storage type for read replica and a flag indicating whether to copy tags to read replica snapshots
     * @param securityParameterGroup     Group of parameters for setting up subnet group for DB instance, name of the IAM role to be used when making API calls to the Directory Service,
     *                                   flag to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts, license model information for the restored DB instance,
     *                                   accessibility options for the DB instance and TDE encryption credentials.
     * @param availabilityZone           The EC2 Availability Zone that the database instance will be created in.
     * @param dbInstanceClass            The compute and memory capacity of the Amazon RDS DB instance.
     * @param dbName                     The database name for the restored DB instance.
     * @param domain                     Specify the Active Directory Domain to restore the instance in.
     * @param engine                     The database engine to use for the new instance.
     * @param iops                       The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for the DB instance.
     * @param multiAZ                    Specifies if the DB instance is a Multi-AZ deployment.
     * @param optionGroupName            The name of the option group to be used for the restored DB instance.
     * @param port                       The port number on which the database accepts connections.
     * @param restoreTime                The date and time to restore from.
     * @param sourceDbInstanceIdentifier The identifier of the source DB instance from which to restore.
     * @param tags                       A list of tags.
     * @param targetDbInstanceIdentifier The name of the new database instance to be created.
     * @param useLatestRestorableTime    Specifies whether (true) or not (false) the DB instance is restored from the latest backup time.
     * @return DBInstance
     */
    public DBInstance restoreDbInstanceToPointInTime(@Config RDSConfiguration config,
                                                     @Connection RDSConnection connection,
                                                     @ParameterGroup(name = "Storage and Maintenance") StorageParameterGroup storageParameterGroup,
                                                     @ParameterGroup(name = "Security") SecurityParameterGroup securityParameterGroup,
                                                     @DisplayName("Availability Zone") @Placement(order = 1) @Optional String availabilityZone,
                                                     @DisplayName("DB Instance Class") @Placement(order = 2) @Optional String dbInstanceClass,
                                                     @DisplayName("DB Name") @Placement(order = 3) @Optional String dbName,
                                                     @Placement(order = 4) @Optional String domain,
                                                     @Placement(order = 5) @Optional String engine,
                                                     @DisplayName("IOPS") @Placement(order = 6) @Optional Integer iops,
                                                     @DisplayName("Multi AZ") @Placement(order = 7) @Optional(defaultValue = "false") boolean multiAZ,
                                                     @DisplayName("Option Group Name") @Placement(order = 8) @Optional String optionGroupName,
                                                     @Placement(order = 9) @Optional Integer port,
                                                     @DisplayName("Restore Time") @Placement(order = 10) @Optional LocalDateTime restoreTime,
                                                     @DisplayName("Source DB Instance Identifier") @Placement(order = 11) String sourceDbInstanceIdentifier,
                                                     @Placement(order = 12) @Optional List<Tag> tags,
                                                     @DisplayName("Target DB Instance Identifier") @Placement(order = 13) String targetDbInstanceIdentifier,
                                                     @DisplayName("Use Latest Restorable Time") @Placement(order = 14) @Optional boolean useLatestRestorableTime) {
        // FIXME: POST GA: Adapt to Atlantic Commons.
        return new DBInstanceServiceImpl(config, connection).restoreDbInstanceToPointInTime(storageParameterGroup.isAutoMinorVersionUpgrade(),
                availabilityZone,
                storageParameterGroup.isCopyTagsToSnapshot(),
                dbInstanceClass,
                dbName,
                securityParameterGroup.getDbSubnetGroupName(),
                domain,
                securityParameterGroup.getDomainIAMRoleName(),
                securityParameterGroup.isEnableIAMDatabaseAuthentication(),
                engine,
                iops,
                securityParameterGroup.getLicenseModel(),
                multiAZ,
                optionGroupName,
                port,
                securityParameterGroup.isPubliclyAccessible(),
                restoreTime,
                sourceDbInstanceIdentifier,
                storageParameterGroup.getStorageType(),
                tags,
                targetDbInstanceIdentifier,
                securityParameterGroup.getTdeCredentialArn(),
                securityParameterGroup.getTdeCredentialPassword(),
                useLatestRestorableTime);
    }

    /**
     * Creates a new DB instance that acts as a Read Replica for an existing source DB instance. You can create a Read
     * Replica for a DB instance running MySQL, MariaDB, or PostgreSQL.
     * Amazon Aurora does not support this action. You must call the CreateDBInstance action to create a DB instance
     * for an Aurora DB cluster.
     * <p><a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstanceReadReplica.html>API Reference</a></p>
     *
     * @param config                          Configuration for RDS connector.
     * @param connection                      Amazon RDS Client connection instance.
     * @param storageParameterGroup           Group of parameters for setting up auto minor version upgrade, storage type for read replica and a flag indicating whether to copy tags to read replica snapshots
     * @param publiclyAccessible              Specifies the accessibility options for the DB instance.
     * @param dbSubnetGroupName               Specifies a DB subnet group for the DB instance.
     * @param kmsKeyId                        The AWS KMS key ID for an encrypted Read Replica.
     * @param preSignedUrl                    The URL that contains a Signature Version 4 signed request for the create-db-instance-read-replica API action in the source AWS Region that contains the source DB instance
     * @param enableIamDatabaseAuthentication True to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts; otherwise false.
     * @param monitoringParameterGroup        Group of parameters for setting up the monitoring interval (in seconds) and ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs
     * @param dbInstanceIdentifier            The DB instance identifier of the Read Replica.
     * @param sourceDbInstanceIdentifier      The identifier of the DB instance that will act as the source for the Read Replica.
     * @param dbInstanceClass                 The compute and memory capacity of the Read Replica.
     * @param availabilityZone                The Amazon EC2 Availability Zone that the Read Replica will be created in.
     * @param port                            The port number that the DB instance uses for connections.
     * @param iops                            The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for the DB instance.
     * @param optionGroupName                 The option group the DB instance will be associated with.
     * @param tags                            A list of tags.
     * @param sourceRegion                    The ID of the region that contains the source for the read replica.
     * @return DBInstance
     */
    public DBInstance createDbInstanceReadReplica(@Config RDSConfiguration config,
                                                  @Connection RDSConnection connection,
                                                  @ParameterGroup(name = STORAGE_AND_MAINTENANCE) StorageParameterGroup storageParameterGroup,
                                                  @DisplayName("Publicly Accessible") @Placement(tab = SECURITY, order = 1) @Optional(defaultValue = "false") boolean publiclyAccessible,
                                                  @DisplayName("DB Subnet Group Name") @Placement(tab = SECURITY, order = 2) @Optional String dbSubnetGroupName,
                                                  @DisplayName("KMS Key ID") @Placement(tab = SECURITY, order = 3) @Optional String kmsKeyId,
                                                  @DisplayName("Pre-Signed URL") @Placement(tab = SECURITY, order = 4) @Optional String preSignedUrl,
                                                  @DisplayName("Enable IAM Database Authentication") @Placement(tab = SECURITY, order = 5) @Optional(defaultValue = "false") boolean enableIamDatabaseAuthentication,
                                                  @ParameterGroup(name = "Monitoring") MonitoringParameterGroup monitoringParameterGroup,
                                                  @DisplayName("DB Instance Identifier") String dbInstanceIdentifier,
                                                  @DisplayName("Source DB Instance Identifier") String sourceDbInstanceIdentifier,
                                                  @DisplayName("DB Instance Class") @Placement(order = 1) @Optional String dbInstanceClass,
                                                  @DisplayName("Availability Zone") @Placement(order = 2) @Optional String availabilityZone,
                                                  @Placement(order = 3) @Optional Integer port,
                                                  @DisplayName("IOPS") @Placement(order = 4) @Optional Integer iops,
                                                  @DisplayName("Option Group Name") @Placement(order = 5) @Optional String optionGroupName,
                                                  @Placement(order = 6) @Optional List<Tag> tags,
                                                  @DisplayName("Source Region") @Placement(order = 7) @Optional String sourceRegion) {
        // FIXME: POST GA: Adapt to Atlantic Commons.
        return new DBInstanceServiceImpl(config, connection).createDBInstanceReadReplica(dbInstanceIdentifier,
                sourceDbInstanceIdentifier,
                dbInstanceClass,
                availabilityZone,
                port,
                storageParameterGroup.isAutoMinorVersionUpgrade(),
                iops,
                optionGroupName,
                publiclyAccessible,
                tags,
                dbSubnetGroupName,
                storageParameterGroup.getStorageType(),
                storageParameterGroup.isCopyTagsToSnapshot(),
                monitoringParameterGroup.getMonitoringInterval(),
                monitoringParameterGroup.getMonitoringRoleArn(),
                kmsKeyId,
                preSignedUrl,
                enableIamDatabaseAuthentication,
                sourceRegion);
    }
}
