/**
 * (c) 2003-2019 MuleSoft, Inc. The software in this package is
 * published under the terms of the Commercial Free Software license V.1, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.extension.rds.internal.operation;

import com.amazonaws.services.rds.model.DescribeDBSnapshotsRequest;
import org.mule.extension.rds.api.attributes.RequestIDAttribute;
import org.mule.extension.rds.api.model.DBSnapshot;
import org.mule.extension.rds.api.model.Filter;
import org.mule.extension.rds.api.model.Tag;
import org.mule.extension.rds.internal.config.RDSConfiguration;
import org.mule.extension.rds.internal.connection.RDSConnection;
import org.mule.extension.rds.internal.error.RDSErrorTypeProvider;
import org.mule.extension.rds.internal.operation.paging.DescribeDBSnapshotsPagingProvider;
import org.mule.extension.rds.internal.service.DBSnapshotService;
import org.mule.extension.rds.internal.service.DBSnapshotServiceImpl;
import org.mule.extension.rds.internal.util.RDSModelFactory;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import java.util.Collection;
import java.util.List;

@Throws(RDSErrorTypeProvider.class)
public class DBSnapshotOperations extends RDSOperations<DBSnapshotService> {


    public DBSnapshotOperations() {
        super(DBSnapshotServiceImpl::new);
    }

    /**
     * <p>
     * Creates a DBSnapshot. The source DBInstance must be in "available" state.
     * </p>
     * <p>
     * <a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBSnapshot.html>API Reference</a>
     * </p>
     *
     * @param config               Configuration for RDS connector.
     * @param client               Amazon RDS Client connection instance.
     * @param dbInstanceIdentifier The DB instance identifier. This is the unique key that identifies a DB instance.
     * @param dbSnapshotIdentifier The identifier for the DB snapshot.
     * @param tags                 A list of tags.
     * @return DBSnapshot
     */
    public DBSnapshot createDbSnapshot(@Config RDSConfiguration config,
                                       @Connection RDSConnection client,
                                       @DisplayName("DB Instance Identifier") String dbInstanceIdentifier,
                                       @DisplayName("DB Snapshot Identifier") String dbSnapshotIdentifier,
                                       @Optional List<Tag> tags) {
        return newExecutionBuilder(config, client).execute(DBSnapshotService::createDBSnapshot)
                .withParam(dbInstanceIdentifier)
                .withParam(dbSnapshotIdentifier)
                .withParam(tags);
    }

    /**
     * <p>
     * Deletes a DBSnapshot. If the snapshot is being copied, the copy operation is terminated.
     * </p>
     * <p>
     * <a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DeleteDBSnapshot.html>API Refernce</a>
     * </p>
     *
     * @param config               Configuration for RDS connector.
     * @param connection           Amazon RDS Client connection instance.
     * @param dbSnapshotIdentifier The DBSnapshot identifier.
     * @return DBSnapshot
     */
    public DBSnapshot deleteDbSnapshot(@Config RDSConfiguration config,
                                       @Connection RDSConnection connection,
                                       @DisplayName("DB Snapshot Identifier") String dbSnapshotIdentifier) {
        return newExecutionBuilder(config, connection).execute(DBSnapshotService::deleteDBSnapshot)
                .withParam(dbSnapshotIdentifier);
    }

    /**
     * <p>
     * Updates a manual DB snapshot, which can be encrypted or not encrypted, with a new engine version.
     * You can update the engine version to either a new major or minor engine version.
     * </p>
     * <p>
     * <a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_ModifyDBSnapshot.html>API Reference</a>
     * </p>
     *
     * @param config               Configuration for RDS connector.
     * @param connection               Amazon RDS Client connection instance.
     * @param dbSnapshotIdentifier The identifier of the DB snapshot to modify.
     * @param engineVersion        The engine version to update the DB snapshot to.
     * @return DBSnapshot
     */
    public DBSnapshot modifyDbSnapshot(@Config RDSConfiguration config,
                                       @Connection RDSConnection connection,
                                       @DisplayName("DB Snapshot Identifier") String dbSnapshotIdentifier,
                                       @DisplayName("Engine Version") @Optional String engineVersion) {
        return newExecutionBuilder(config, connection).execute(DBSnapshotService::modifyDBSnapshot)
                .withParam(dbSnapshotIdentifier)
                .withParam(engineVersion);
    }

    /**
     * <p>
     * Returns information about DB snapshots. This API action supports pagination.
     * </p>
     * <p>
     * <a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBSnapshots.html>API Reference</a>
     * </p>
     *
     * @param dbInstanceIdentifier The ID of the DB instance to retrieve the list of DB snapshots for.
     * @param dbSnapshotIdentifier A specific DB snapshot identifier to describe. This parameter cannot be used in conjunction with DBInstanceIdentifier.
     * @param includePublic        Set this value to true to include manual DB snapshots that are public and can be copied or restored by any AWS account, otherwise set this value to false. The default is false.
     * @param includeShared        Set this value to true to include shared manual DB snapshots from other AWS accounts that this AWS account has been given permission to copy or restore, otherwise set this value to false. The default is false.
     * @param snapshotType         The type of snapshots to be returned.
     * @param maxRecords           The maximum number of records to include in the response.
     * @param filters              This parameter is not currently supported.
     * @return DescribeDBSnapshotsResult
     */

    public PagingProvider<RDSConnection, Result<DBSnapshot, RequestIDAttribute>> describeDbSnapshots(@DisplayName("DB Instance Identifier") @Optional String dbInstanceIdentifier,
                                                                                                     @DisplayName("DB Snapshot Identifier") @Optional String dbSnapshotIdentifier,
                                                                                                     @DisplayName("Include Public") @Optional(defaultValue = "false") boolean includePublic,
                                                                                                     @DisplayName("Include Shared") @Optional(defaultValue = "false") boolean includeShared,
                                                                                                     @DisplayName("Snapshot Type") @Optional String snapshotType,
                                                                                                     @DisplayName("Max Records") @Optional Integer maxRecords,
                                                                                                     @Optional Collection<Filter> filters) {

        return new DescribeDBSnapshotsPagingProvider(new DescribeDBSnapshotsRequest()
                .withMaxRecords(java.util.Optional.ofNullable(maxRecords).orElse(20))
                .withDBInstanceIdentifier(dbInstanceIdentifier)
                .withDBSnapshotIdentifier(dbSnapshotIdentifier)
                .withIncludePublic(includePublic)
                .withIncludeShared(includeShared)
                .withSnapshotType(snapshotType)
                .withFilters(RDSModelFactory.unWrapFilterList(filters)));
    }
}
