/**
 * (c) 2003-2019 MuleSoft, Inc. The software in this package is
 * published under the terms of the Commercial Free Software license V.1, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.extension.rds.internal.operation;

import com.amazonaws.services.rds.model.DescribeEventsRequest;
import org.mule.connectors.commons.template.service.ConnectorService;
import org.mule.extension.rds.api.attributes.RequestIDAttribute;
import org.mule.extension.rds.api.model.Event;
import org.mule.extension.rds.internal.connection.RDSConnection;
import org.mule.extension.rds.internal.error.RDSErrorTypeProvider;
import org.mule.extension.rds.internal.operation.paging.DescribeEventsPagingProvider;
import org.mule.extension.rds.internal.util.DateUtils;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import java.time.LocalDateTime;
import java.util.List;

@Throws(RDSErrorTypeProvider.class)
public class EventOperations extends RDSOperations<ConnectorService> {

    public EventOperations() {
        super(null);
    }

    /**
     * Returns events related to DB instances, DB security groups, DB snapshots, and DB parameter groups for the
     * past 14 days. Events specific to a particular DB instance, DB security group, database snapshot, or DB parameter
     * group can be obtained by providing the name as a parameter. By default, the past hour of events are returned.
     * <p> <a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeEvents.html>API Reference</a></p>
     *
     * @param duration         The number of minutes to retrieve events for.
     * @param endTime          The end of the time interval for which to retrieve events, specified in ISO 8601 format.
     * @param eventCategories  A list of event categories that trigger notifications for a event notification subscription.
     * @param sourceIdentifier The identifier of the event source for which events will be returned. If not specified, then all sources are included in the response.
     * @param sourceType       The event source to retrieve events for. If no value is specified, all events are returned.
     * @param startTime        The beginning of the time interval to retrieve events for, specified in ISO 8601 format.
     * @return DescribeEventsResult
     */
    public PagingProvider<RDSConnection, Result<Event, RequestIDAttribute>> describeEvents(@Optional Integer duration,
                                                                                           @DisplayName("End Time") @Optional LocalDateTime endTime,
                                                                                           @DisplayName("Event Categories") @Optional List<String> eventCategories,
                                                                                           @DisplayName("Source Identifier") @Optional String sourceIdentifier,
                                                                                           @DisplayName("Source Type") @Optional String sourceType,
                                                                                           @DisplayName("Start Time") @Optional LocalDateTime startTime) {
        return new DescribeEventsPagingProvider(new DescribeEventsRequest()
                .withDuration(duration)
                .withEndTime(DateUtils.toDate(endTime))
                .withEventCategories(eventCategories)
                .withSourceIdentifier(sourceIdentifier)
                .withSourceType(sourceType)
                .withStartTime(DateUtils.toDate(startTime)));
    }
}
