/**
 * (c) 2003-2019 MuleSoft, Inc. The software in this package is
 * published under the terms of the Commercial Free Software license V.1, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.extension.rds.internal.operation;

import com.amazonaws.services.rds.model.DescribeReservedDBInstancesOfferingsRequest;
import com.amazonaws.services.rds.model.DescribeReservedDBInstancesRequest;
import org.mule.extension.rds.api.attributes.RequestIDAttribute;
import org.mule.extension.rds.api.model.ReservedDBInstance;
import org.mule.extension.rds.api.model.ReservedDBInstancesOffering;
import org.mule.extension.rds.api.model.Tag;
import org.mule.extension.rds.internal.config.RDSConfiguration;
import org.mule.extension.rds.internal.connection.RDSConnection;
import org.mule.extension.rds.internal.error.RDSErrorTypeProvider;
import org.mule.extension.rds.internal.operation.paging.DescribeReservedDBInstancesPagingProvider;
import org.mule.extension.rds.internal.operation.paging.DescribeReservedDbInstancesOfferingsPagingProvider;
import org.mule.extension.rds.internal.service.ReservedDBInstanceService;
import org.mule.extension.rds.internal.service.ReservedDBInstanceServiceImpl;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import java.util.List;

@Throws(RDSErrorTypeProvider.class)
public class ReservedDBInstanceOperations extends RDSOperations<ReservedDBInstanceService> {

    public ReservedDBInstanceOperations() {
        super(ReservedDBInstanceServiceImpl::new);
    }

    /**
     * Returns information about reserved DB instances for this account, or about a specified reserved DB instance.
     * <p>
     * <a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstances.html>API Reference</a>
     * </p>
     *
     * @param reservedDbInstanceId          The reserved DB instance identifier filter value.
     * @param reservedDbInstancesOfferingId The offering identifier filter value. Specify this parameter to show only purchased reservations matching the specified offering identifier.
     * @param dbInstanceClass               The DB instance class filter value. Specify this parameter to show only those reservations matching the specified DB instances class.
     * @param duration                      The duration filter value, specified in years or seconds. Specify this parameter to show only reservations for this duration.
     * @param productDescription            The product description filter value. Specify this parameter to show only those reservations matching the specified product description.
     * @param offeringType                  The offering type filter value. Specify this parameter to show only the available offerings matching the specified offering type.
     * @param multiAZ                       The Multi-AZ filter value. Specify this parameter to show only those reservations matching the specified Multi-AZ parameter.
     * @return DescribeReservedDBInstancesResult
     */
    public PagingProvider<RDSConnection, Result<ReservedDBInstance, RequestIDAttribute>> describeReservedDbInstances(
            @DisplayName("Reserved DB Instance ID") @Optional String reservedDbInstanceId,
            @DisplayName("Reserved DB Instances Offering ID") @Optional String reservedDbInstancesOfferingId,
            @DisplayName("DB Instance Class") @Optional String dbInstanceClass,
            @Optional String duration,
            @DisplayName("Product Description") @Optional String productDescription,
            @DisplayName("Offering Type") @Optional String offeringType,
            @DisplayName("Multi AZ") @Optional(defaultValue = "false") boolean multiAZ) {
        DescribeReservedDBInstancesRequest describeReservedDBInstancesRequest = new DescribeReservedDBInstancesRequest()
                .withReservedDBInstanceId(reservedDbInstanceId)
                .withReservedDBInstancesOfferingId(reservedDbInstancesOfferingId)
                .withDBInstanceClass(dbInstanceClass)
                .withDuration(duration)
                .withProductDescription(productDescription)
                .withOfferingType(offeringType)
                .withMultiAZ(multiAZ);
        return new DescribeReservedDBInstancesPagingProvider(describeReservedDBInstancesRequest);
    }

    /**
     * Lists available reserved DB instance offerings.
     * <p><a href=http://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeReservedDBInstancesOfferings.html>API Reference</a></p>
     *
     * @param dbInstanceClass               The DB instance class filter value. Specify this parameter to show only the available offerings matching the specified DB instance class.
     * @param duration                      Duration filter value, specified in years or seconds. Specify this parameter to show only reservations for this duration.
     * @param multiAZ                       The Multi-AZ filter value. Specify this parameter to show only the available offerings matching the specified Multi-AZ parameter.
     * @param offeringType                  The offering type filter value. Specify this parameter to show only the available offerings matching the specified offering type.
     * @param productDescription            Product description filter value. Specify this parameter to show only the available offerings matching the specified product description.
     * @param reservedDbInstancesOfferingId The offering identifier filter value. Specify this parameter to show only the available offering that matches the specified reservation identifier.
     * @return DescribeReservedDBInstancesOfferingsResult
     */
    public PagingProvider<RDSConnection, Result<ReservedDBInstancesOffering, RequestIDAttribute>> describeReservedDbInstancesOfferings(
            @DisplayName("DB Instance Class") @Optional String dbInstanceClass,
            @Optional String duration,
            @DisplayName("Multi AZ") @Optional(defaultValue = "false") boolean multiAZ,
            @DisplayName("Offering Type") @Optional String offeringType,
            @DisplayName("Product Description") @Optional String productDescription,
            @DisplayName("Reserved DB Instances Offering ID") @Optional String reservedDbInstancesOfferingId) {
        DescribeReservedDBInstancesOfferingsRequest describeReservedDBInstancesOfferingsRequest = new DescribeReservedDBInstancesOfferingsRequest()
                .withDBInstanceClass(dbInstanceClass)
                .withDuration(duration)
                .withMultiAZ(multiAZ)
                .withOfferingType(offeringType)
                .withProductDescription(productDescription)
                .withReservedDBInstancesOfferingId(reservedDbInstancesOfferingId);
        return new DescribeReservedDbInstancesOfferingsPagingProvider(describeReservedDBInstancesOfferingsRequest);
    }

    /**
     * Purchases a reserved DB instance offering.
     * <p><a href=https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_PurchaseReservedDBInstancesOffering.html>API Reference</a></p>
     *
     * @param config                        Configuration for RDS connector.
     * @param connection                        Amazon RDS Client connection instance.
     * @param reservedDbInstancesOfferingId The ID of the Reserved DB instance offering to purchase.
     * @param dbInstanceCount               The number of instances to reserve.
     * @param reservedDbInstanceId          Customer-specified identifier to track this reservation.
     * @param tags                          A list of tags.
     * @return ReservedDBInstance
     */
    public ReservedDBInstance purchaseReservedDbInstancesOffering(@Config RDSConfiguration config,
                                                                  @Connection RDSConnection connection,
                                                                  @DisplayName("Reserved DB Instances Offering ID") String reservedDbInstancesOfferingId,
                                                                  @DisplayName("DB Instance Count") @Optional Integer dbInstanceCount,
                                                                  @DisplayName("Reserved DB Instance ID") @Optional String reservedDbInstanceId,
                                                                  @Optional List<Tag> tags) {
        return newExecutionBuilder(config, connection).execute(ReservedDBInstanceService::purchaseReservedDBInstancesOffering)
                .withParam(reservedDbInstancesOfferingId)
                .withParam(dbInstanceCount)
                .withParam(reservedDbInstanceId)
                .withParam(tags);
    }
}
