/**
 * (c) 2003-2019 MuleSoft, Inc. The software in this package is
 * published under the terms of the Commercial Free Software license V.1, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.extension.rds.internal.operation.paging;

import com.amazonaws.AmazonWebServiceResult;
import com.amazonaws.services.rds.AmazonRDS;
import org.mule.extension.rds.api.attributes.RequestIDAttribute;
import org.mule.extension.rds.internal.connection.RDSConnection;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.stream.Collectors.toList;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;

public abstract class RDSPagingProvider<TYPE, UNPARSED_TYPE, REQUEST, RESULT extends AmazonWebServiceResult> implements PagingProvider<RDSConnection, Result<TYPE, RequestIDAttribute>> {

    private final BiFunction<AmazonRDS, REQUEST, RESULT> clientListFunction;
    private final BiFunction<REQUEST, String, REQUEST> appendMarkerFunction;
    private final Supplier<RESULT> resultConstructor;
    private final Function<RESULT, List<UNPARSED_TYPE>> extractionFunction;
    private final Function<UNPARSED_TYPE, TYPE> parsingFunction;
    private final REQUEST request;
    private final Function<RESULT, String> getMarkerFunction;
    private Function<AmazonRDS, RESULT> resultRetrievalFunction;

    public RDSPagingProvider(REQUEST request,
                             BiFunction<AmazonRDS, REQUEST, RESULT> clientListFunction,
                             Function<RESULT, String> getMarkerFunction,
                             BiFunction<REQUEST, String, REQUEST> appendMarkerFunction,
                             Supplier<RESULT> resultConstructor,
                             Function<RESULT, List<UNPARSED_TYPE>> extractionFunction,
                             Function<UNPARSED_TYPE, TYPE> parsingFunction) {
        this.request = request;
        this.resultRetrievalFunction = client -> clientListFunction.apply(client, request);
        this.clientListFunction = clientListFunction;
        this.getMarkerFunction = getMarkerFunction;
        this.appendMarkerFunction = appendMarkerFunction;
        this.resultConstructor = resultConstructor;
        this.extractionFunction = extractionFunction;
        this.parsingFunction = parsingFunction;
    }

    private List<Result<TYPE, RequestIDAttribute>> getBatch(AmazonRDS initialClient) {
        RESULT result = resultRetrievalFunction.apply(initialClient);
        resultRetrievalFunction = Optional.ofNullable(getMarkerFunction.apply(result))
                .<Function<AmazonRDS, RESULT>>map(token ->
                        client -> this.clientListFunction.apply(client, appendMarkerFunction.apply(request, token)))
                .orElseGet(() -> value -> resultConstructor.get());
        return extractionFunction.apply(result)
                .stream()
                .map(parsingFunction)
                .map(parsedValue -> Result.<TYPE, RequestIDAttribute>builder()
                        .output(parsedValue)
                        .attributes(new RequestIDAttribute(result.getSdkResponseMetadata().getRequestId()))
                        .mediaType(APPLICATION_JAVA)
                        .build())
                .collect(toList());
    }


    @Override
    public List<Result<TYPE, RequestIDAttribute>> getPage(RDSConnection rdsClient) {
        return getBatch(rdsClient.getAwsClient());
    }

    @Override
    public Optional<Integer> getTotalResults(RDSConnection rdsClient) {
        return null;
    }

    @Override
    public void close(RDSConnection rdsClient) throws MuleException {
        // Do nothing.
    }
}



