/**
 * (c) 2003-2019 MuleSoft, Inc. The software in this package is
 * published under the terms of the Commercial Free Software license V.1, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */
package org.mule.extension.rds.internal.service;


import com.amazonaws.services.rds.AmazonRDS;
import com.amazonaws.services.rds.model.CreateDBInstanceReadReplicaRequest;
import com.amazonaws.services.rds.model.CreateDBInstanceRequest;
import com.amazonaws.services.rds.model.DeleteDBInstanceRequest;
import com.amazonaws.services.rds.model.ModifyDBInstanceRequest;
import com.amazonaws.services.rds.model.RebootDBInstanceRequest;
import com.amazonaws.services.rds.model.RestoreDBInstanceFromDBSnapshotRequest;
import com.amazonaws.services.rds.model.RestoreDBInstanceToPointInTimeRequest;
import com.amazonaws.services.rds.model.StartDBInstanceRequest;
import com.amazonaws.services.rds.model.StopDBInstanceRequest;
import org.mule.extension.rds.api.model.DBInstance;
import org.mule.extension.rds.api.model.Tag;
import org.mule.extension.rds.internal.config.RDSConfiguration;
import org.mule.extension.rds.internal.connection.RDSConnection;
import org.mule.extension.rds.internal.operation.group.MonitoringParameterGroup;
import org.mule.extension.rds.internal.operation.group.dbinstance.CreationDBInstanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.dbinstance.ModificationDBInstanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.security.CreationSecurityParameterGroup;
import org.mule.extension.rds.internal.operation.group.security.ModificationSecurityParameterGroup;
import org.mule.extension.rds.internal.operation.group.storage.CreationStorageAndMaintenanceParameterGroup;
import org.mule.extension.rds.internal.operation.group.storage.ModificationStorageAndMaintenanceParameterGroup;
import org.mule.extension.rds.internal.util.DateUtils;
import org.mule.extension.rds.internal.util.RDSModelFactory;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

public class DBInstanceServiceImpl extends RDSServiceImpl implements DBInstanceService {

    public DBInstanceServiceImpl(RDSConfiguration config, RDSConnection connection) {
        super(config, connection);
    }

    @Override
    public DBInstance createDbInstance(CreationStorageAndMaintenanceParameterGroup storageAndMaintenanceParameterGroup,
                                       CreationSecurityParameterGroup securityParameterGroup,
                                       MonitoringParameterGroup monitoringParameterGroup,
                                       CreationDBInstanceParameterGroup dbInstanceParameterGroup) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().createDBInstance(new CreateDBInstanceRequest()
                .withAllocatedStorage(storageAndMaintenanceParameterGroup.getAllocatedStorage())
                .withAutoMinorVersionUpgrade(storageAndMaintenanceParameterGroup.isAutoMinorVersionUpgrade())
                .withBackupRetentionPeriod(storageAndMaintenanceParameterGroup.getBackupRetentionPeriod())
                .withCopyTagsToSnapshot(storageAndMaintenanceParameterGroup.isCopyTagsToSnapshot())
                .withDBClusterIdentifier(storageAndMaintenanceParameterGroup.getDbClusterIdentifier())
                .withPreferredBackupWindow(storageAndMaintenanceParameterGroup.getPreferredBackupWindow())
                .withPreferredMaintenanceWindow(storageAndMaintenanceParameterGroup.getPreferredMaintenanceWindow())
                .withPromotionTier(storageAndMaintenanceParameterGroup.getPromotionTier())
                .withStorageEncrypted(storageAndMaintenanceParameterGroup.isStorageEncrypted())
                .withStorageType(storageAndMaintenanceParameterGroup.getStorageType())
                .withDBSecurityGroups(securityParameterGroup.getDbSecurityGroups())
                .withDBSubnetGroupName(securityParameterGroup.getDbSubnetGroupName())
                .withDomainIAMRoleName(securityParameterGroup.getDomainIAMRoleName())
                .withEnableIAMDatabaseAuthentication(securityParameterGroup.isEnableIAMDatabaseAuthentication())
                .withKmsKeyId(securityParameterGroup.getKmsKeyId())
                .withLicenseModel(securityParameterGroup.getLicenseModel())
                .withMasterUsername(securityParameterGroup.getMasterUsername())
                .withMasterUserPassword(securityParameterGroup.getMasterUserPassword())
                .withPubliclyAccessible(securityParameterGroup.isPubliclyAccessible())
                .withTdeCredentialArn(securityParameterGroup.getTdeCredentialArn())
                .withTdeCredentialPassword(securityParameterGroup.getTdeCredentialPassword())
                .withVpcSecurityGroupIds(securityParameterGroup.getVpcSecurityGroupIds())
                .withMonitoringInterval(monitoringParameterGroup.getMonitoringInterval())
                .withMonitoringRoleArn(monitoringParameterGroup.getMonitoringRoleArn())
                .withAvailabilityZone(dbInstanceParameterGroup.getAvailabilityZone())
                .withCharacterSetName(dbInstanceParameterGroup.getCharacterSetName())
                .withDBInstanceClass(dbInstanceParameterGroup.getDbInstanceClass())
                .withDBInstanceIdentifier(dbInstanceParameterGroup.getDbInstanceIdentifier())
                .withDBParameterGroupName(dbInstanceParameterGroup.getDbParameterGroupName())
                .withDBName(dbInstanceParameterGroup.getDbName())
                .withPort(dbInstanceParameterGroup.getPort())
                .withDomain(dbInstanceParameterGroup.getDomain())
                .withEngine(dbInstanceParameterGroup.getEngine())
                .withEngineVersion(dbInstanceParameterGroup.getEngineVersion())
                .withIops(dbInstanceParameterGroup.getIops())
                .withMultiAZ(dbInstanceParameterGroup.isMultiAZ())
                .withOptionGroupName(dbInstanceParameterGroup.getOptionGroupName())
                .withTags(RDSModelFactory.wrapTagList(dbInstanceParameterGroup.getTags()))
                .withTimezone(dbInstanceParameterGroup.getTimezone())));
    }

    @Override
    public DBInstance modifyDbInstance(ModificationStorageAndMaintenanceParameterGroup storageAndMaintenanceParameterGroup,
                                       ModificationSecurityParameterGroup securityParameterGroup,
                                       MonitoringParameterGroup monitoringParameterGroup,
                                       ModificationDBInstanceParameterGroup dbInstanceParameterGroup) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().modifyDBInstance(new ModifyDBInstanceRequest()
                .withAllocatedStorage(storageAndMaintenanceParameterGroup.getAllocatedStorage())
                .withAllowMajorVersionUpgrade(storageAndMaintenanceParameterGroup.isAllowMajorVersionUpgrade())
                .withApplyImmediately(storageAndMaintenanceParameterGroup.isApplyImmediately())
                .withAutoMinorVersionUpgrade(storageAndMaintenanceParameterGroup.isAutoMinorVersionUpgrade())
                .withBackupRetentionPeriod(storageAndMaintenanceParameterGroup.getBackupRetentionPeriod())
                .withCopyTagsToSnapshot(storageAndMaintenanceParameterGroup.isCopyTagsToSnapshot())
                .withPreferredBackupWindow(storageAndMaintenanceParameterGroup.getPreferredBackupWindow())
                .withPreferredMaintenanceWindow(storageAndMaintenanceParameterGroup.getPreferredMaintenanceWindow())
                .withPromotionTier(storageAndMaintenanceParameterGroup.getPromotionTier())
                .withStorageType(storageAndMaintenanceParameterGroup.getStorageType())
                .withCACertificateIdentifier(securityParameterGroup.getCaCertificateIdentifier())
                .withDBSecurityGroups(securityParameterGroup.getDbSecurityGroups())
                .withDBSubnetGroupName(securityParameterGroup.getDbSubnetGroupName())
                .withDomainIAMRoleName(securityParameterGroup.getDomainIAMRoleName())
                .withEnableIAMDatabaseAuthentication(securityParameterGroup.isEnableIAMDatabaseAuthentication())
                .withLicenseModel(securityParameterGroup.getLicenseModel())
                .withMasterUserPassword(securityParameterGroup.getMasterUserPassword())
                .withPubliclyAccessible(securityParameterGroup.isPubliclyAccessible())
                .withTdeCredentialArn(securityParameterGroup.getTdeCredentialArn())
                .withTdeCredentialPassword(securityParameterGroup.getTdeCredentialPassword())
                .withVpcSecurityGroupIds(securityParameterGroup.getVpcSecurityGroupIds())
                .withMonitoringInterval(monitoringParameterGroup.getMonitoringInterval())
                .withMonitoringRoleArn(monitoringParameterGroup.getMonitoringRoleArn())
                .withDBInstanceClass(dbInstanceParameterGroup.getDbInstanceClass())
                .withDBInstanceIdentifier(dbInstanceParameterGroup.getDbInstanceIdentifier())
                .withDBParameterGroupName(dbInstanceParameterGroup.getDbParameterGroupName())
                .withDBPortNumber(dbInstanceParameterGroup.getDbPortNumber())
                .withDomain(dbInstanceParameterGroup.getDomain())
                .withEngineVersion(dbInstanceParameterGroup.getEngineVersion())
                .withIops(dbInstanceParameterGroup.getIops())
                .withMultiAZ(dbInstanceParameterGroup.isMultiAZ())
                .withNewDBInstanceIdentifier(dbInstanceParameterGroup.getNewDBInstanceIdentifier())
                .withOptionGroupName(dbInstanceParameterGroup.getOptionGroupName())));
    }

    @Override
    public DBInstance deleteDBInstance(String dBInstanceIdentifier,
                                       String finalDBSnapshotIdentifier,
                                       Boolean skipFinalSnapshot) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().deleteDBInstance(new DeleteDBInstanceRequest()
                .withDBInstanceIdentifier(dBInstanceIdentifier)
                .withFinalDBSnapshotIdentifier(finalDBSnapshotIdentifier)
                .withSkipFinalSnapshot(skipFinalSnapshot)));
    }

    @Override
    public DBInstance startDBInstance(String dBInstanceIdentifier) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().startDBInstance(new StartDBInstanceRequest().withDBInstanceIdentifier(dBInstanceIdentifier)));
    }


    @Override
    public DBInstance stopDBInstance(String dBInstanceIdentifier,
                                     String dBSnapshotIdentifier) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().stopDBInstance(new StopDBInstanceRequest()
                .withDBInstanceIdentifier(dBInstanceIdentifier)
                .withDBSnapshotIdentifier(dBSnapshotIdentifier)));
    }

    @Override
    public DBInstance rebootDBInstance(String dBInstanceIdentifier,
                                       Boolean forceFailover) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().rebootDBInstance(new RebootDBInstanceRequest()
                .withDBInstanceIdentifier(dBInstanceIdentifier)
                .withForceFailover(forceFailover)));
    }

    @Override
    public DBInstance restoreDbInstanceToPointInTime(boolean autoMinorVersionUpgrade,
                                                     String availabilityZone,
                                                     boolean copyTagsToSnapshot,
                                                     String dBInstanceClass,
                                                     String dBName,
                                                     String dBSubnetGroupName,
                                                     String domain,
                                                     String domainIAMRoleName,
                                                     boolean enableIAMDatabaseAuthentication,
                                                     String engine,
                                                     Integer iops,
                                                     String licenseModel,
                                                     boolean multiAZ,
                                                     String optionGroupName,
                                                     Integer port,
                                                     boolean publiclyAccessible,
                                                     LocalDateTime restoreTime,
                                                     String sourceDBInstanceIdentifier,
                                                     String storageType,
                                                     Collection<Tag> tags,
                                                     String targetDBInstanceIdentifier,
                                                     String tdeCredentialArn,
                                                     String tdeCredentialPassword,
                                                     boolean useLatestRestorableTime) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().restoreDBInstanceToPointInTime(new RestoreDBInstanceToPointInTimeRequest()
                .withAutoMinorVersionUpgrade(autoMinorVersionUpgrade)
                .withAvailabilityZone(availabilityZone)
                .withCopyTagsToSnapshot(copyTagsToSnapshot)
                .withDBInstanceClass(dBInstanceClass)
                .withDBName(dBName)
                .withDBSubnetGroupName(dBSubnetGroupName)
                .withDomain(domain)
                .withDomainIAMRoleName(domainIAMRoleName)
                .withEnableIAMDatabaseAuthentication(enableIAMDatabaseAuthentication)
                .withEngine(engine)
                .withIops(iops)
                .withLicenseModel(licenseModel)
                .withMultiAZ(multiAZ)
                .withOptionGroupName(optionGroupName)
                .withPort(port)
                .withPubliclyAccessible(publiclyAccessible)
                .withRestoreTime(DateUtils.toDate(restoreTime))
                .withSourceDBInstanceIdentifier(sourceDBInstanceIdentifier)
                .withStorageType(storageType)
                .withTargetDBInstanceIdentifier(targetDBInstanceIdentifier)
                .withTags(RDSModelFactory.wrapTagList(tags))
                .withTdeCredentialArn(tdeCredentialArn)
                .withTdeCredentialPassword(tdeCredentialPassword)
                .withUseLatestRestorableTime(useLatestRestorableTime)));
    }


    public DBInstance restoreDbInstanceFromDbSnapshot(boolean autoMinorVersionUpgrade,
                                                      String availabilityZone,
                                                      boolean copyTagsToSnapshot,
                                                      String dBInstanceClass,
                                                      String dBInstanceIdentifier,
                                                      String dBSnapshotIdentifier,
                                                      String dBName,
                                                      String dBSubnetGroupName,
                                                      String domain,
                                                      String domainIAMRoleName,
                                                      boolean enableIAMDatabaseAuthentication,
                                                      String engine,
                                                      Integer iops,
                                                      String licenseModel,
                                                      boolean multiAZ,
                                                      String optionGroupName,
                                                      Integer port,
                                                      boolean publiclyAccessible,
                                                      String storageType,
                                                      Collection<Tag> tags,
                                                      String tdeCredentialArn,
                                                      String tdeCredentialPassword) {
        return RDSModelFactory.wrapDBInstance(getAwsClient().restoreDBInstanceFromDBSnapshot(new RestoreDBInstanceFromDBSnapshotRequest()
                .withDBInstanceIdentifier(dBInstanceIdentifier)
                .withDBSnapshotIdentifier(dBSnapshotIdentifier)
                .withAutoMinorVersionUpgrade(autoMinorVersionUpgrade)
                .withAvailabilityZone(availabilityZone)
                .withCopyTagsToSnapshot(copyTagsToSnapshot)
                .withDBInstanceClass(dBInstanceClass)
                .withDBName(dBName)
                .withDBSubnetGroupName(dBSubnetGroupName)
                .withDomain(domain)
                .withDomainIAMRoleName(domainIAMRoleName)
                .withEnableIAMDatabaseAuthentication(enableIAMDatabaseAuthentication)
                .withEngine(engine).withMultiAZ(multiAZ)
                .withIops(iops).withLicenseModel(licenseModel)
                .withOptionGroupName(optionGroupName).withPubliclyAccessible(publiclyAccessible)
                .withStorageType(storageType).withPort(port)
                .withTags(RDSModelFactory.wrapTagList(tags))
                .withTdeCredentialArn(tdeCredentialArn)
                .withTdeCredentialPassword(tdeCredentialPassword)));
    }

    public DBInstance createDBInstanceReadReplica(String dbInstanceIdentifier,
                                                  String sourceDbInstanceIdentifier,
                                                  String dbInstanceClass,
                                                  String availabilityZone,
                                                  Integer port,
                                                  boolean autoMinorVersionUpgrade,
                                                  Integer iops,
                                                  String optionGroupName,
                                                  boolean publiclyAccessible,
                                                  List<Tag> tags,
                                                  String dbSubnetGroupName,
                                                  String storageType,
                                                  boolean copyTagsToSnapshot,
                                                  Integer monitoringInterval,
                                                  String monitoringRoleArn,
                                                  String kmsKeyId,
                                                  String preSignedUrl,
                                                  boolean enableIamDatabaseAuthentication,
                                                  String sourceRegion) {
        return RDSModelFactory.wrapCreateDBInstanceReadReplica(getAwsClient().createDBInstanceReadReplica(new CreateDBInstanceReadReplicaRequest()
                .withDBInstanceIdentifier(dbInstanceIdentifier)
                .withSourceDBInstanceIdentifier(sourceDbInstanceIdentifier)
                .withDBInstanceClass(dbInstanceClass)
                .withAvailabilityZone(availabilityZone)
                .withPort(port)
                .withAutoMinorVersionUpgrade(autoMinorVersionUpgrade)
                .withIops(iops)
                .withOptionGroupName(optionGroupName)
                .withPubliclyAccessible(publiclyAccessible)
                .withTags(RDSModelFactory.wrapTagList(tags))
                .withDBSubnetGroupName(dbSubnetGroupName)
                .withStorageType(storageType)
                .withCopyTagsToSnapshot(copyTagsToSnapshot)
                .withMonitoringInterval(monitoringInterval)
                .withMonitoringRoleArn(monitoringRoleArn)
                .withKmsKeyId(kmsKeyId)
                .withPreSignedUrl(preSignedUrl)
                .withEnableIAMDatabaseAuthentication(enableIamDatabaseAuthentication)
                .withSourceRegion(sourceRegion)));
    }

    protected AmazonRDS getAwsClient() {
        return getConnection().getAwsClient();
    }
}
