package org.mule.connectors.commons.template.metadata;

import org.mule.connectors.atlantic.commons.builder.clazz.ClassBuilder;
import org.mule.connectors.atlantic.commons.builder.execution.ExecutionBuilder;
import org.mule.connectors.atlantic.commons.builder.lambda.function.BiFunction;
import org.mule.connectors.commons.template.config.ConnectorConfig;
import org.mule.connectors.commons.template.connection.ConnectorConnection;
import org.mule.connectors.commons.template.service.ConnectorService;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MetadataContext;

import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import static java.lang.String.format;

/**
 * Parent metadata resolver for all Metadata Resolvers on the connectors. Provides an easy access to an ExecutionBuilder.
 *
 * @param <CONFIG>     The config to be used.
 * @param <CONNECTION> The connection of the connector.
 * @param <SERVICE>    The metadata service.
 */
public class ConnectorMetadataResolver<CONFIG extends ConnectorConfig, CONNECTION extends ConnectorConnection, SERVICE extends ConnectorService> {

    private final BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructor;

    /**
     * Default constructor. This constructor should be called with the Default constructor of the Service.
     *
     * @param serviceConstructor The constructor of the service implementation.
     */
    protected ConnectorMetadataResolver(BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructor) {
        this.serviceConstructor = serviceConstructor;
    }

    protected ConnectorMetadataResolver(Class<SERVICE> serviceClass) {
        this(serviceClass, clazz -> {
            try {
                return (Class<SERVICE>) Class.forName(format("%sImpl", serviceClass.getName()));
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
        });
    }

    protected ConnectorMetadataResolver(Class<SERVICE> serviceClass, Function<Class<SERVICE>, Class<? extends SERVICE>> transformer) {
        this((config, connection) -> (SERVICE) Stream.of(Optional.of(serviceClass)
                .filter(Class::isInterface)
                .map(transformer)
                .orElse(serviceClass)
                .getDeclaredConstructors())
                .filter(constructor -> ConnectorConfig.class.isAssignableFrom(constructor.getParameterTypes()[0]) && ConnectorConnection.class.isAssignableFrom(constructor.getParameterTypes()[1]))
                .findFirst()
                .orElseThrow(NoSuchMethodException::new)
                .newInstance(config, connection));
    }

    protected ExecutionBuilder<SERVICE> newExecutionBuilder(MetadataContext metadataContext) throws ConnectionException {
        return new ClassBuilder().create(serviceConstructor)
                .withParam(metadataContext.<CONFIG>getConfig().get())
                .withParam(metadataContext.<CONNECTION>getConnection().get());
    }
}
