package org.mule.connectors.commons.template.operation;

import org.mule.connectors.atlantic.commons.builder.clazz.ClassBuilder;
import org.mule.connectors.atlantic.commons.builder.config.ConfigurableBuilder;
import org.mule.connectors.atlantic.commons.builder.execution.ExecutionBuilder;
import org.mule.connectors.atlantic.commons.builder.lambda.function.BiFunction;
import org.mule.connectors.commons.template.config.ConnectorConfig;
import org.mule.connectors.commons.template.connection.ConnectorConnection;
import org.mule.connectors.commons.template.service.ConnectorService;

import java.lang.reflect.Constructor;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import static java.lang.String.format;

/**
 * Parent class for Certified Connector Operations that provides with a factory method that allows the fluent execution
 * of the methods of the service.
 *
 * @param <CONFIG>     The config object that handles this operation.
 * @param <CONNECTION> The type of connection to be handled by these operations.
 * @param <SERVICE>    The service that handles the operations connection to the remote system.
 */
public class ConnectorOperations<CONFIG extends ConnectorConfig, CONNECTION extends ConnectorConnection, SERVICE extends ConnectorService> {

    private final BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructor;

    /**
     * Default constructor. This constructor should be called with the
     *
     * @param serviceConstructor
     */
    protected ConnectorOperations(BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructor) {
        this.serviceConstructor = serviceConstructor;
    }

    protected ConnectorOperations(Class<SERVICE> serviceClass) {
        this(serviceClass, clazz -> {
            try {
                return (Class<SERVICE>)Class.forName(format("%sImpl", serviceClass.getName()));
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
        });
    }

    protected ConnectorOperations(Class<SERVICE> serviceClass, Function<Class<SERVICE>, Class<? extends SERVICE>> transformer) {
        this((config, connection) -> (SERVICE)Stream.of(Optional.of(serviceClass)
                .filter(Class::isInterface)
                .map(transformer)
                .orElse(serviceClass)
                .getDeclaredConstructors())
                .filter(constructor -> ConnectorConfig.class.isAssignableFrom(constructor.getParameterTypes()[0]) && ConnectorConnection.class.isAssignableFrom(constructor.getParameterTypes()[1]))
                .findFirst()
                .orElseThrow(NoSuchMethodException::new)
                .newInstance(config, connection));
    }

    protected ExecutionBuilder<SERVICE> newExecutionBuilder(CONFIG config, CONNECTION connection) {
        return new ClassBuilder().create(serviceConstructor)
                .withParam(config)
                .withParam(connection);
    }
}
