package com.mulesoft.connectors.commons.template.metadata;

import com.mulesoft.connectors.commons.template.config.ConnectorConfig;
import com.mulesoft.connectors.commons.template.connection.ConnectorConnection;
import com.mulesoft.connectors.commons.template.service.BlockingConnectorService;
import org.mule.commons.atlantic.Atlantic;
import org.mule.commons.atlantic.exception.UnhandledException;
import org.mule.commons.atlantic.execution.builder.factory.InstanceExecutionBuilderFactory;
import org.mule.commons.atlantic.lambda.function.BiFunction;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;

import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import static java.lang.String.format;
import static org.mule.commons.atlantic.Atlantic.newStaticExecutionBuilder;

/**
 * Parent metadata resolver for all Metadata Resolvers on the connectors. Provides an easy access to an ExecutionBuilder.
 *
 * @param <CONFIG>     The config to be used.
 * @param <CONNECTION> The connection of the connector.
 * @param <SERVICE>    The metadata service.
 */
public class ConnectorMetadataResolver<CONFIG extends ConnectorConfig, CONNECTION extends ConnectorConnection, SERVICE extends BlockingConnectorService, ERROR extends Enum<ERROR> & ErrorTypeDefinition<ERROR>> {

    private final BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructor;
    private final ERROR unknownErrorTypeDefinition;

    /**
     * Default constructor. This constructor should be called with the Default constructor of the Service.
     *
     * @param serviceConstructor         The constructor of the service implementation.
     * @param unknownErrorTypeDefinition The error to throw when an unexpected error occurs.
     */
    protected ConnectorMetadataResolver(BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructor, ERROR unknownErrorTypeDefinition) {
        this.serviceConstructor = serviceConstructor;
        this.unknownErrorTypeDefinition = unknownErrorTypeDefinition;
    }

    /**
     * Constructor that receives only the interface of the {@link BlockingConnectorService} to use. It assumes that there's an implementation named the same but with Impl at the end (i.e. MyService would have MyServiceImpl).
     *
     * @param serviceClass               The interface of the service.
     * @param unknownErrorTypeDefinition The error to throw when an unexpected error occurs.
     */
    protected ConnectorMetadataResolver(Class<SERVICE> serviceClass,
                                        ERROR unknownErrorTypeDefinition) {
        this(serviceClass, clazz -> {
            try {
                return (Class<SERVICE>) Class.forName(format("%sImpl", serviceClass.getName()));
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
        }, unknownErrorTypeDefinition);
    }

    /**
     * Constructor that receives the interface of the {@link BlockingConnectorService} to use and a transformer {@link Function} that converts that class into an implementation one.
     *
     * @param serviceClass               The interface of the service.
     * @param transformer                The transforming function.
     * @param unknownErrorTypeDefinition The error to throw when an unexpected error occurs.
     */
    protected ConnectorMetadataResolver(Class<SERVICE> serviceClass,
                                        Function<Class<SERVICE>, Class<? extends SERVICE>> transformer,
                                        ERROR unknownErrorTypeDefinition) {
        this((config, connection) -> (SERVICE) Stream.of(Optional.of(serviceClass)
                        .filter(Class::isInterface)
                        .map(transformer)
                        .orElse(serviceClass)
                        .getDeclaredConstructors())
                        .filter(constructor -> ConnectorConfig.class.isAssignableFrom(constructor.getParameterTypes()[0]) && ConnectorConnection.class.isAssignableFrom(constructor.getParameterTypes()[1]))
                        .findFirst()
                        .orElseThrow(NoSuchMethodException::new)
                        .newInstance(config, connection),
                unknownErrorTypeDefinition);
    }

    /**
     * Creates and returns an {@link InstanceExecutionBuilderFactory} for an instance of the service used by this resolver based on the information available on the {@link MetadataContext}.
     *
     * @param metadataContext The context of the metadata.
     * @param <RESULT>        The type of object that will be returned by the execution of the service.
     * @return InstanceExecutionBuilderFactory The factory of the method to execute from the service.
     */
    protected <RESULT> InstanceExecutionBuilderFactory<SERVICE, RESULT> newExecutionBuilderFactory(MetadataContext metadataContext) {
        try {
            return Atlantic.<SERVICE, RESULT>newInstanceExecutionBuilder(newStaticExecutionBuilder()
                    .execute(serviceConstructor)
                    .withParam(metadataContext.<CONFIG>getConfig().get())
                    .withParam(metadataContext.<CONNECTION>getConnection().get()))
                    .withIgnoredExceptionType(ModuleException.class)
                    .withExceptionHandler(Throwable.class, t -> new ModuleException(unknownErrorTypeDefinition, t));
        } catch (ConnectionException e) {
            // FIXME: solve this later.
            throw new UnhandledException(e);
        }
    }
}
