package com.mulesoft.connectors.commons.template.metadata;

import org.mule.runtime.api.value.Value;
import org.mule.runtime.extension.api.values.ValueBuilder;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

/**
 * Default {@link ValueProvider} that provides the from a fixed place, whether it's an enum, map, collection or array.
 */
public class FixedValuesValueProvider implements ValueProvider {

    private final Map<String, String> fixedValueMap;

    /**
     * Constructor that defines the contents to be shown from the entries of a {@link Map}.
     *
     * @param fixedValueMap The map that will define the values. The keys are the effective values to be set and the values are the displayed names.
     */
    public FixedValuesValueProvider(Map<String, String> fixedValueMap) {
        this.fixedValueMap = fixedValueMap;
    }

    /**
     * Constructor that defines the contents to be shown from the elements of a {@link Collection}.
     *
     * @param collection The collection that will define the values. The keys and the displayed names are the same and equal to the values of the collection.
     */
    public FixedValuesValueProvider(Collection<String> collection) {
        this.fixedValueMap = collection.stream().collect(toMap(identity(), identity()));
    }

    /**
     * Constructor that defines the contents to be shown from the elements of an {@link Enum}.
     *
     * @param enumValues The enum that will define the values. The keys and the displayed names are the same and equal to the values of the enum.
     */
    public FixedValuesValueProvider(Enum<?>... enumValues) {
        this(Stream.of(enumValues).map(Enum::name).collect(toList()));
    }


    /**
     * Constructor that defines the contents to be shown from the elements of an array of {@link String}s.
     *
     * @param fixedValues The array that will define the values. The keys and the displayed names are the same and equal to the values of the array.
     */
    public FixedValuesValueProvider(String... fixedValues) {
        this(Stream.of(fixedValues).collect(toList()));
    }

    @Override
    public Set<Value> resolve() throws ValueResolvingException {
        return fixedValueMap.entrySet().stream()
                .map(entry -> ValueBuilder.newValue(entry.getKey())
                        .withDisplayName(entry.getValue())
                        .build())
                .collect(toSet());
    }
}
