package com.mulesoft.connectors.commons.template.operation;

import com.mulesoft.connectors.commons.template.config.ConnectorConfig;
import com.mulesoft.connectors.commons.template.connection.ConnectorConnection;
import com.mulesoft.connectors.commons.template.service.ConnectorService;
import org.mule.commons.atlantic.Atlantic;
import org.mule.commons.atlantic.execution.builder.factory.InstanceExecutionBuilderFactory;
import org.mule.commons.atlantic.lambda.function.TriFunction;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;

import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import static java.lang.String.format;
import static org.mule.commons.atlantic.Atlantic.newStaticExecutionBuilder;

/**
 * Parent class for all non-blocking operations. Children of this class should contain a single public method for the
 * operation.
 * <p>
 * Non-blocking operations should not be the default as, normally, there is no non-blocking mechanism unless manually
 * created.
 *
 * @param <CONFIG>     The configuration type of the operation. Must implement {@link ConnectorConfig}.
 * @param <CONNECTION> The connection type of the operation. Must implement {@link ConnectorConnection}.
 * @param <SERVICE>    The service type of the operation. Must extend {@link ConnectorService}.
 * @param <PAYLOAD>    The type of payload returned by the operation. Void if none.
 * @param <ATTRIBUTES> The type of attributes returned by the operation. Void if none.
 * @param <ERROR>      The type of error thrown when an uncaught, unexpected and unknown error occurs.
 */
public class NonBlockingConnectorOperation<CONFIG extends ConnectorConfig, CONNECTION extends ConnectorConnection, SERVICE extends ConnectorService, PAYLOAD, ATTRIBUTES, ERROR extends Enum & ErrorTypeDefinition> {

    private final TriFunction<CONFIG, CONNECTION, CompletionCallback<PAYLOAD, ATTRIBUTES>, SERVICE> serviceConstructorCall;
    private final ERROR unexpectedErrorValue;

    /**
     * Default constructor. Accepts the constructor of the service implementation and the unexpected error type.
     * Service constructor must be a 3 parameters constructor that takes a child of {@link ConnectorConfig} as the first
     * parameter and {@link ConnectorConnection} as its second one and a {@link CompletionCallback} as its third.
     *
     * @param serviceConstructorCall The constructor.
     * @param unexpectedErrorValue   The error thrown when an uncaught, unexpected and unknown error occurs.
     */
    public NonBlockingConnectorOperation(TriFunction<CONFIG, CONNECTION, CompletionCallback<PAYLOAD, ATTRIBUTES>, SERVICE> serviceConstructorCall, ERROR unexpectedErrorValue) {
        this.serviceConstructorCall = serviceConstructorCall;
        this.unexpectedErrorValue = unexpectedErrorValue;
    }

    /**
     * Constructor that allows for the custom specification of how the implementation of the passed service class should
     * be retrieved.
     *
     * @param serviceClass         This should be the class of the interface.
     * @param transformer          This is the function that converts from the class of the interface to the class of
     *                             the implementation.
     *                             The implementation should have a 2 parameters constructor that takes a child of
     *                             {@link ConnectorConfig} as the first parameter and {@link ConnectorConnection} as its
     *                             second one.
     * @param unexpectedErrorValue The error thrown when an uncaught, unexpected and unknown error occurs.
     */
    protected NonBlockingConnectorOperation(Class<SERVICE> serviceClass,
                                            Function<Class<SERVICE>, Class<? extends SERVICE>> transformer,
                                            ERROR unexpectedErrorValue) {
        this((config, connection, completionCallback) -> (SERVICE) Stream.of(Optional.of(serviceClass)
                        .filter(Class::isInterface)
                        .map(transformer)
                        .orElse(serviceClass)
                        .getDeclaredConstructors())
                        .filter(constructor -> ConnectorConfig.class.isAssignableFrom(constructor.getParameterTypes()[0]) && ConnectorConnection.class.isAssignableFrom(constructor.getParameterTypes()[1]) && CompletionCallback.class.isAssignableFrom(constructor.getParameterTypes()[2]))
                        .findFirst()
                        .orElseThrow(NoSuchMethodException::new)
                        .newInstance(config, connection, completionCallback),
                unexpectedErrorValue);
    }


    /**
     * Default implementation of ConnectorOperation{@link #NonBlockingConnectorOperation(Class, Function, Enum)} that defines a
     * regular function that converts the class of the interface to the class of the implementation by adding an Impl to
     * its name.
     *
     * @param serviceClass         This should be the class of the interface.
     * @param unexpectedErrorValue The error thrown when an uncaught, unexpected and unknown error occurs.
     */
    protected NonBlockingConnectorOperation(Class<SERVICE> serviceClass,
                                            ERROR unexpectedErrorValue) {
        this(serviceClass, clazz -> {
                    try {
                        return (Class<SERVICE>) Class.forName(format("%sImpl", serviceClass.getName()));
                    } catch (ClassNotFoundException e) {
                        throw new RuntimeException(e);
                    }
                },
                unexpectedErrorValue);
    }


    /**
     * Factory method that allows the creation of an {@link InstanceExecutionBuilderFactory}. This class will allow the
     * execution of all the methods on the service interface declared as a generic on this class and provide them as a
     * fluent executor.
     *
     * @param config     The config of the operation.
     * @param connection The connection of the operation.
     * @param <RESULT>   The type of result expected from the execution.
     * @return InstanceExecutionBuilderFactory The factory.
     */
    protected <RESULT> InstanceExecutionBuilderFactory<SERVICE, RESULT> newExecutionBuilderFactory(CONFIG config,
                                                                                                   CONNECTION connection,
                                                                                                   CompletionCallback<PAYLOAD, ATTRIBUTES> completionCallback) {
        return Atlantic.<SERVICE, RESULT>newInstanceExecutionBuilder(
                newStaticExecutionBuilder().execute(serviceConstructorCall)
                        .withParam(config)
                        .withParam(connection)
                        .withParam(completionCallback))
                .withIgnoredExceptionType(ModuleException.class)
                .withExceptionHandler(Throwable.class, t -> {
                    throw new ModuleException(unexpectedErrorValue, t);
                });
    }
}
