package com.mulesoft.connectors.http.commons.connection.provider;

import com.mulesoft.connectors.commons.template.connection.provider.ConnectorConnectionProvider;
import com.mulesoft.connectors.http.commons.connection.ConnectorHttpConnection;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.RefName;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.proxy.BaseProxyConfigBuilder;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;

import javax.inject.Inject;
import java.util.stream.Collectors;

import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.extension.api.annotation.param.ParameterGroup.ADVANCED;
import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;

public abstract class AbstractHttpConnectionProvider<C extends ConnectorHttpConnection> implements ConnectorConnectionProvider<C> {

    @RefName
    private String configName;

    /**
     * The maximum number of outbound connections that will be kept open at the same time. By default the number of connections is
     * unlimited.
     */
    @Parameter
    @Optional(defaultValue = "-1")
    @Expression(NOT_SUPPORTED)
    @Placement(tab = ADVANCED_TAB, order = 3)
    @DisplayName("Maximum outbound connections")
    @Summary("The maximum number of outbound connections that will be kept open at the same time.")
    private Integer maxConnections;

    /**
     * The number of milliseconds that a connection can remain idle before it is closed. The value of this attribute is only used
     * when persistent connections are enabled.
     */
    @Parameter
    @Optional(defaultValue = "30000")
    @Expression(NOT_SUPPORTED)
    @Placement(tab = ADVANCED_TAB, order = 4)
    @DisplayName("Connection idle timeout")
    @Summary("The number of milliseconds that a connection can remain idle before it is closed.")
    private Integer connectionIdleTimeout;

    @Parameter
    @Optional(defaultValue = "30000")
    @Expression(NOT_SUPPORTED)
    @Placement(tab = ADVANCED_TAB, order = 6)
    @DisplayName("Connection timeout (ms)")
    @Summary("Defines how long to wait for the outbound connection to be created.")
    private Integer connectionTimeout;

    @Optional
    @Parameter
    @DisplayName("Proxy")
    @Summary("Proxy Configuration")
    @Placement(tab = ADVANCED, order = 3)
    private ProxyConfiguration proxyConfig;

    @Inject
    private HttpService httpService;

    public C connect() throws ConnectionException {
        HttpClient httpClient = httpService.getClientFactory().create(new HttpClientConfiguration.Builder()
                .setProxyConfig(java.util.Optional.ofNullable(proxyConfig)
                        .map(proxyConfiguration -> java.util.Optional.ofNullable(proxyConfiguration.getNtlmDomain())
                        .<BaseProxyConfigBuilder>map(ProxyConfig.NtlmProxyConfig.builder()::ntlmDomain)
                        .orElseGet(ProxyConfig::builder)
                        .host(proxyConfiguration.getHost())
                        .port(proxyConfiguration.getPort())
                        .nonProxyHosts(proxyConfiguration.getNonProxyHosts().stream()
                                .collect(Collectors.joining(",")))
                        .username(proxyConfiguration.getUsername())
                        .password(proxyConfiguration.getPassword())
                        .build())
                        .orElse(null))
                .setClientSocketProperties(TcpClientSocketProperties.builder()
                        .connectionTimeout(connectionTimeout)
                        .build())
                .setMaxConnections(maxConnections)
                .setConnectionIdleTimeout(connectionIdleTimeout)
                .setName(configName)
                .build());
        httpClient.start();
        return connect(httpClient);
    }

    public HttpClientConfiguration.Builder customizeHttpClient(HttpClientConfiguration.Builder builder) {
        return builder;
    }

    public abstract C connect(HttpClient httpClient) throws ConnectionException;
}
