package com.mulesoft.connectors.http.commons.operation;

import com.mulesoft.connectors.commons.template.operation.BlockingConnectorPagedOperation;
import com.mulesoft.connectors.commons.template.operation.pagination.OffsetPaginationContext;
import com.mulesoft.connectors.http.commons.config.HttpConnectorConfig;
import com.mulesoft.connectors.http.commons.connection.ConnectorHttpConnection;
import com.mulesoft.connectors.http.commons.service.HttpClientService;
import com.mulesoft.connectors.http.commons.transform.Transformer;

import org.mule.commons.atlantic.execution.builder.factory.InstanceExecutionBuilderFactory;
import org.mule.commons.atlantic.lambda.function.BiFunction;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class GetPaginatedOperation<
        CONFIG extends HttpConnectorConfig,
        CONNECTION extends ConnectorHttpConnection,
        SERVICE extends HttpClientService<CONFIG, CONNECTION>>
        extends BlockingConnectorPagedOperation<CONFIG, CONNECTION, SERVICE, OffsetPaginationContext> {

    private final ErrorTypeDefinition<?> transformationErrorValue;
    private final Transformer<InputStream, Map<String, Object>> inputStreamToMapTransformer;
    private Integer totalCount = 0;
    private final static String RESPONSE_FIELD_TOTAL_COUNT = "total_count";
    private final static String RESPONSE_FIELD_ENTRIES_LIST = "entries";

    protected GetPaginatedOperation(
            BiFunction<CONFIG, CONNECTION, SERVICE> serviceConstructorCall,
            ErrorTypeDefinition<?> unexpectedErrorValue,
            ErrorTypeDefinition<?> transformationErrorValue,
            Transformer<InputStream, Map<String, Object>> inputStreamToMapTransformer) {
        super(serviceConstructorCall, unexpectedErrorValue);
        this.transformationErrorValue = transformationErrorValue;
        this.inputStreamToMapTransformer = inputStreamToMapTransformer;
    }

    @Override
    protected <RESULT> List<RESULT> getPage(
        InstanceExecutionBuilderFactory<SERVICE, RESULT> executionBuilder,
        OffsetPaginationContext pagingContext) {

        HttpResponse httpResponse = executionBuilder
            .execute(SERVICE::get)
            .withParam(pagingContext.getRelativePath())
            .withParam(pagingContext.getPathParams())
            .withParam(pagingContext.getQueryParams())
            .withParam(pagingContext.getHeaders())
            .withParam(new InputStreamHttpEntity(pagingContext.getRequestBody()));

        pagingContext.nextPage();

        try {
            InputStream responseStream = this.getResponseStream(httpResponse);
            Map<String, Object> responseObject = this.inputStreamToMapTransformer.transform(responseStream);
            int totalResults = Integer.parseInt(responseObject.get(RESPONSE_FIELD_TOTAL_COUNT).toString());
            pagingContext.setTotalResults(totalResults);

            return (List<RESULT>) responseObject.get(RESPONSE_FIELD_ENTRIES_LIST);
        } catch (IOException e) {
            throw new ModuleException(String.format("IOException while reading response. Exception: %s.", e), this.transformationErrorValue);
        }
    }

    @Override
    protected <RESULT> Optional<Integer> getTotalResults(InstanceExecutionBuilderFactory<SERVICE, RESULT> executionBuilder, OffsetPaginationContext pagingContext) {
        return Optional.of(pagingContext.getTotalResults());
    }

    private InputStream getResponseStream(HttpResponse method) throws IOException {
        // get response input stream
        InputStream result = null;
        HttpEntity responseEntity = method.getEntity();
        if (responseEntity != null) {
            result = responseEntity.getContent();
        }

        if (result == null || result.available() == 0) {
            throw new IOException("Empty stream came as response");
        }

        return result;
    }
}
