/*
 * Decompiled with CFR 0.152.
 */
package org.mule.extension.redis.internal.connection;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.mule.extension.redis.internal.connection.RedisConnection;
import org.mule.extension.redis.internal.connection.RedisObjectStoreExpiryTask;
import org.mule.extension.redis.internal.error.exceptions.InvalidDataException;
import org.mule.extension.redis.internal.service.RedisAPIService;
import org.mule.extension.redis.internal.service.factory.ServiceFactory;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.serialization.ObjectSerializer;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreNotAvailableException;
import org.mule.runtime.api.store.ObjectStoreSettings;
import org.mule.runtime.api.store.TemplateObjectStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RedisObjectStore
extends TemplateObjectStore<Serializable> {
    private static final Logger logger = LoggerFactory.getLogger(RedisObjectStore.class);
    private final RedisAPIService service;
    private final Integer entryTTL;
    private final Integer maxEntries;
    private final long expirationInterval;
    private ObjectSerializer objectSerializer;
    private Scheduler scheduler;
    private ScheduledFuture<?> scheduledTask;
    private String name;

    public RedisObjectStore(RedisConnection redisConnection, ObjectStoreSettings settings, String name) throws ObjectStoreNotAvailableException {
        if (redisConnection.isValid()) {
            ServiceFactory serviceFactory = new ServiceFactory();
            this.service = serviceFactory.getService(redisConnection);
            this.entryTTL = this.computeEntryTTL(redisConnection, settings);
            this.maxEntries = settings.getMaxEntries().orElse(null);
            if (!settings.isPersistent()) {
                logger.warn("You are trying to configure a transient object store on a system that is persistent. The PERSISTENT setting of the Object Store will be ignored");
            }
        } else {
            throw new ObjectStoreNotAvailableException(I18nMessageFactory.createStaticMessage((String)"Cannot access Redis instance"));
        }
        this.name = name;
        this.expirationInterval = settings.getExpirationInterval();
        this.objectSerializer = redisConnection.getObjectSerializer();
        this.scheduler = redisConnection.getScheduler();
    }

    protected boolean doContains(String key) throws ObjectStoreException {
        try {
            return this.service.exists(key);
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public boolean isPersistent() {
        return true;
    }

    public void clear() throws ObjectStoreException {
        try {
            String[] keys = (String[])this.service.keys("*").stream().map(String::new).distinct().toArray(String[]::new);
            if (keys.length != 0) {
                this.service.del(keys);
            }
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public void open() throws ObjectStoreException {
        if (this.expirationInterval > 0L) {
            try {
                this.scheduledTask = this.scheduler.scheduleWithFixedDelay((Runnable)new RedisObjectStoreExpiryTask(this), 0L, this.expirationInterval, TimeUnit.MILLISECONDS);
            }
            catch (Exception e) {
                throw new ObjectStoreException(I18nMessageFactory.createStaticMessage((String)("ObjectStore expiry task could not be scheduled for object store: " + this.name)), (Throwable)e);
            }
        }
    }

    public void close() {
        if (this.scheduledTask != null) {
            this.scheduledTask.cancel(true);
        }
    }

    public List<String> allKeys() throws ObjectStoreException {
        try {
            return this.service.keys("*").stream().map(String::new).collect(Collectors.toList());
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public Map<String, Serializable> retrieveAll() throws ObjectStoreException {
        try {
            return this.service.keys("*").stream().map(String::new).collect(Collectors.toMap(Function.identity(), k -> this.fromByteArray(this.service.get((String)k))));
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    protected void doStore(String key, Serializable value) throws ObjectStoreException {
        try {
            this.service.set(key, this.toByteArray(value), this.entryTTL, false);
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public Serializable doRetrieve(String key) throws ObjectStoreException {
        try {
            return this.fromByteArray(this.service.get(key));
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    protected Serializable doRemove(String key) throws ObjectStoreException {
        try {
            return this.service.del(key);
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public RedisAPIService getService() {
        return this.service;
    }

    public Integer getMaxEntries() {
        return this.maxEntries;
    }

    public long getExpirationInterval() {
        return this.expirationInterval;
    }

    private byte[] toByteArray(Serializable serializable) {
        return this.objectSerializer.getInternalProtocol().serialize((Object)serializable);
    }

    private Serializable fromByteArray(byte[] bytes) {
        return (Serializable)this.objectSerializer.getInternalProtocol().deserialize(bytes);
    }

    private Integer computeEntryTTL(RedisConnection redisConnection, ObjectStoreSettings settings) {
        Integer ttl = null;
        if (settings.getEntryTTL().isPresent()) {
            ttl = Math.toIntExact((Long)settings.getEntryTTL().get() / 1000L);
        }
        if (redisConnection.getEntryTTL() != null) {
            if (ttl != null && ttl > redisConnection.getEntryTTL()) {
                throw new InvalidDataException("The entry TTL set for the custom object store is bigger than the one set in the Connection configuration");
            }
            ttl = redisConnection.getEntryTTL() / 1000;
        }
        return ttl;
    }
}

