/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.api;

import com.mulesoft.connectors.mqtt3.internal.routing.MQTT3Message;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Parameter;

import java.io.Serializable;
import java.util.Objects;

import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

/**
 * Represents an {@link MQTT3Message}'s attributes.
 *
 * @since 1.0
 */

public class MQTT3MessageAttributes implements Serializable {

  private static final long serialVersionUID = 4913835689071707740L;
  /**
   * The topic to which the message was published.
   */
  @Parameter
  private String topic;

  /**
   * The MQTT id that identifies a message received from the broker.
   */
  @Parameter
  private int messageId;

  /**
   * The quality of service with which the message was delivered by the broker.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  private int qos;

  /**
   * If set to true, this message might be a duplicate of one which has already been received.
   */
  @Parameter
  private boolean isDuplicate;

  /**
   * If set to true, this message was either sent from a current publisher, or was "retained" by the server
   * as the last message published on the topic.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  private boolean isRetained;

  /**
   * Returns an instance of MQTT3MessageAttributes.
   * @param topic the message's topic.
   * @param id the message's id.
   * @param qos the message's quality of service.
   * @param isDuplicate whether the message could be a duplicate of another one that was already received.
   * @param isRetained whether the message was retained by the broker.
   */
  private MQTT3MessageAttributes(String topic, int id, int qos, boolean isDuplicate, boolean isRetained) {
    this.topic = topic;
    this.messageId = id;
    this.qos = qos;
    this.isDuplicate = isDuplicate;
    this.isRetained = isRetained;
  }

  /**
   * The MQTT3MessageAttributes default constructor.
   * @return an instance of MQTT3MessageAttributes.
   */
  public MQTT3MessageAttributes() {}

  /**
   * @return the message's topic
   */
  public String getTopic() {
    return topic;
  }

  /**
   * @return the message's id.
   */
  public int getMessageId() {
    return messageId;
  }

  /**
   * The message's quality of service which will be the minimum
   * between the message's original qos level for publish and
   * the listeners qos level for subscription.
   * @return the message's quality of service.
   */
  public int getQos() {
    return qos;
  }

  /**
   * @return true if this message might be a duplicate of one which has already been received.
   */
  public boolean isDuplicate() {
    return isDuplicate;
  }

  /**
   * @return true if this message was either sent from a current publisher, or was "retained" by the server,
   * as the last message published on the topic.
   */
  public boolean isRetained() {
    return isRetained;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if ((o == null) || o.getClass() != this.getClass()) {
      return false;
    }

    MQTT3MessageAttributes otherAttr = (MQTT3MessageAttributes) o;
    return new EqualsBuilder().append(topic, otherAttr.topic).append(messageId, otherAttr.messageId).append(qos, otherAttr.qos)
        .append(isRetained, otherAttr.isRetained).append(isDuplicate, otherAttr.isDuplicate).isEquals();
  }

  @Override
  public int hashCode() {
    return Objects.hash(topic, messageId, qos, isRetained, isDuplicate);
  }

  /**
   * Builder implementation for creating a {@link MQTT3MessageAttributes} instance.
   *
   * @since 1.0
   */
  public static class Builder {

    private String topic;
    private int messageId;
    private int qos;
    private boolean duplicate;
    private boolean isRetained;

    private Builder() {}

    /**
     * @return a new builder instance.
     */
    public static Builder newInstance() {
      return new Builder();
    }

    /**
     * Sets a topic for the {@link MQTT3MessageAttributes} instance.
     * @param topic the message's topic
     * @return the updated builder instance.
     */
    public Builder withTopic(String topic) {
      this.topic = topic;
      return this;
    }

    /**
     * Sets a qos for the {@link MQTT3MessageAttributes} instance.
     * @param qos the message's qos
     * @return the updated builder instance.
     */
    public Builder withQoS(int qos) {
      this.qos = qos;
      return this;
    }

    /**
     * Sets a message id for the {@link MQTT3MessageAttributes} instance.
     * @param id the message's id
     * @return the updated builder instance.
     */
    public Builder withMessageId(int id) {
      this.messageId = id;
      return this;
    }

    /**
     * Sets the retained flag for the {@link MQTT3MessageAttributes} instance.
     * @param isRetained the message's retained flag.
     * @return the updated builder instance.
     */
    public Builder withRetained(boolean isRetained) {
      this.isRetained = isRetained;
      return this;
    }

    /**
     * Sets the duplicate flag for the {@link MQTT3MessageAttributes} instance.
     * @param isDuplicate the message's duplicate flag.
     * @return the updated builder instance.
     */
    public Builder withDuplicate(boolean isDuplicate) {
      this.duplicate = isDuplicate;
      return this;
    }

    /**
     * @return a new {@link MQTT3MessageAttributes} instance.
     */
    public MQTT3MessageAttributes build() {
      return new MQTT3MessageAttributes(topic, messageId, qos, duplicate, isRetained);
    }

  }
}
