/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.api.clientId;

import com.mulesoft.connectors.mqtt3.internal.exceptions.MQTT3InvalidClientIDException;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.sdk.api.annotation.semantics.security.ClientId;
import org.slf4j.Logger;

import java.util.Objects;

import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * A {@link ClientIDGenerator} implementation that allows the user
 * to specify an client id and a custom expression to be
 * added to the client id as a suffix. Both are optional, but at least one
 * of these fields must be specified in order to successfully generate a client ID.
 */
@Summary("Generates a client id from a static prefix and a dynamic suffix")
public class ClientIDCustomExpressionGenerator implements ClientIDGenerator {

  private static final Logger LOGGER = getLogger(ClientIDCustomExpressionGenerator.class);

  /**
   * Client id used to identify the connection to the MQTT broker.
   */
  @Parameter
  @DisplayName("Client ID")
  @Optional(defaultValue = "")
  @Expression(NOT_SUPPORTED)
  @ClientId
  @Summary("A static client id to identify the connection to the broker")
  protected String clientId;

  /**
   * Custom client id suffix used to identify the connection to the MQTT broker.
   */
  @Parameter
  @DisplayName("Custom Client ID Suffix")
  @Expression(SUPPORTED)
  @Optional(defaultValue = "")
  @Summary("A dynamic client id suffix")
  protected String customClientIdSuffix;

  /**
   * @return the client ID that results from concatenating the clientId and customClientIdSuffix parameters.
   */
  @Override
  public String generateClientID() {

    if (clientId.isEmpty() && customClientIdSuffix.isEmpty()) {
      LOGGER
          .error("A custom expression client id generator was used, but neither the 'Client ID' parameter nor the 'Custom Client ID Suffix' parameter has been specified.");
      throw new MQTT3InvalidClientIDException("A custom expression client id generator was used, but neither the 'Client ID' parameter nor the 'Custom Client ID Suffix' parameter has been specified.");
    }

    return clientId + customClientIdSuffix;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof ClientIDCustomExpressionGenerator)) {
      return false;
    }

    ClientIDCustomExpressionGenerator that = (ClientIDCustomExpressionGenerator) o;

    return clientId.equals(that.clientId) && customClientIdSuffix.equals(that.customClientIdSuffix);
  }

  @Override
  public int hashCode() {
    return Objects.hash(clientId, customClientIdSuffix);
  }
}
