/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.api.clientId;

import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.sdk.api.annotation.semantics.security.ClientId;

import java.util.Objects;
import java.util.UUID;

import static java.util.UUID.randomUUID;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;


/**
 * A {@link ClientIDGenerator} implementation that will attach a random UUID to
 * the specified Client ID, as a suffix. If a client ID is not provided, the random
 * UUID will be used as the connection identifier.
 */
@Summary("Generates a client id from a static prefix and appends a random uuid suffix")
public class ClientIDRandomSuffixGenerator implements ClientIDGenerator {

  /**
   * Client id to be used to identify the connection to the MQTT broker.
   */
  @Parameter
  @DisplayName("Client ID")
  @Optional(defaultValue = "")
  @Expression(NOT_SUPPORTED)
  @ClientId
  @Summary("A static client id prefix to identify the connection to the broker")
  protected String clientId;

  private UUID randomSuffix = randomUUID();

  /**
   * @return a client ID that results from concatenating the clientId and a randomly generated UUID string.
   */
  @Override
  public String generateClientID() {
    return clientId + (clientId.isEmpty() ? "" : "-") + randomSuffix;
  }

  public String getClientId() {
    return this.clientId;
  }

  public void setClientId(String clientId) {
    this.clientId = clientId;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof ClientIDRandomSuffixGenerator)) {
      return false;
    }

    ClientIDRandomSuffixGenerator that = (ClientIDRandomSuffixGenerator) o;

    return clientId.equals(that.clientId) && randomSuffix.equals(that.randomSuffix);
  }

  @Override
  public int hashCode() {
    return Objects.hash(clientId, randomSuffix);
  }
}
