/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection;

import com.mulesoft.connectors.mqtt3.api.MQTT3MessageAttributes;
import com.mulesoft.connectors.mqtt3.api.Topic;
import com.mulesoft.connectors.mqtt3.internal.exceptions.MQTT3PublishException;
import com.mulesoft.connectors.mqtt3.internal.routing.MQTT3MessageHandler;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;

import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * Provides a way to interact with mqtt connection objects.
 *
 * @since 1.0
 */
public interface MQTT3Connection {

  /**
   * Registers a {@code MQTT3MessageHandler} locally for the provided topics without subscribing to the broker.
   * This allows handlers to be ready before connecting to prevent race conditions.
   * @param topics the list of {@code Topic} filters for which the handler will be registered.
   * @param messageHandler a {@code MQTT3MessageHandler} to be called when messages for these topics are received.
   */
  void registerMessageHandler(List<Topic> topics, MQTT3MessageHandler messageHandler);

  /**
   * Subscribes to the provided topics on the broker. The message handlers should already be registered.
   * @param topics the list of {@code Topic} filters to subscribe to on the broker.
   * @throws ConnectionException if an error occurs during subscription.
   */
  void subscribeToTopics(List<Topic> topics) throws ConnectionException;

  /**
   * Establishes the connection to the MQTT broker.
   * @throws ConnectionException if connection attempt was unsuccessful.
   */
  void connect() throws ConnectionException;

  /**
   * De-registers the provided {@code MQTT3MessageHandler} from the list of provided topics.
   * For the topics which have no remaining {@code MQTT3MessageHandler} associated, the connection will
   * unsubscribe.
   * @param topics the list of {@code Topic} filters for which the {@code MQTT3MessageHandler} will be removed.
   * @param messageHandler the {@code MQTT3MessageHandler} will be removed.
   */
  void unsubscribeListenerFromTopics(List<Topic> topics, MQTT3MessageHandler messageHandler);

  /**
   * Publishes a message to an mqtt topic.
   * @param topic the topic to which the message should be published.
   * @param message the message content to be sent.
   * @param qos the quality of service with which the message should be published.
   * @param retained whether the message should be retained by the broker.
   * @return an Integer {@link CompletableFuture} that will return the messageId of the published message on success.
   * @throws MQTT3PublishException if the publish operation fails.
   * @throws ConnectionException if a connectivity error arises during publish.
   */
  CompletableFuture<Integer> publish(String topic, byte[] message, int qos, boolean retained)
      throws MQTT3PublishException, ConnectionException;

  /**
   * Sets the TLS context for the connection.
   * @param tlsContextFactory the {@link TlsContextFactory} with the required data.
   * @throws ConnectionException
   */
  void setTLSOptions(TlsContextFactory tlsContextFactory) throws ConnectionException;

  /**
   * Sets the username and password for the connection to authenticate to the mqtt broker.
   * @param username the authentication username.
   * @param password the authentication password.
   */
  void setUsernamePassword(String username, String password);

  /**
   * Sets a list of fail-over servers to iterate over and attempt to establish a connection.
   * @param failOverServerArray the list of servers to iterate over.
   */
  void setFailOverServers(String[] failOverServerArray);

  /**
   * @return true if the broker has persisted a session for a client with the same client id, or false if there's no session present.
   */
  boolean isSessionPresent();

  /**
   * @return true if connection is still active.
   */
  boolean isConnected();

  /**
   * Tests if the connection can be established without changing the current connection state.
   * This is useful for validation purposes during redeployments.
   * @return true if the connection can be established successfully.
   * @throws ConnectionException if the connection cannot be established.
   */
  boolean testConnectivity() throws ConnectionException;

  void addSourceCallbackToConnectionLostHandler(SourceCallback<byte[], MQTT3MessageAttributes> callback);

  /**
   * Perform client disconnect.
   */
  void disconnect();
}
