/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

import static org.mule.runtime.extension.api.annotation.param.display.Placement.ADVANCED_TAB;

/**
 * Represents an MQTT connection's settings.
 *
 * @since 1.0
 */
public class MQTT3ConnectionOptions {

  /**
   * The socket connection timeout value. This attribute works in tandem with {@link #timeoutUnit}.
   * In the case of multiple fail-over URLs provided, the timeout applies to each one individually.
   */
  @Parameter
  @Optional(defaultValue = "30")
  @Placement(tab = ADVANCED_TAB, order = 1)
  @Summary("Socket connection timeout value")
  private int connectionTimeout;

  /**
   * A {@link TimeUnit} which qualifies the {@link #connectionTimeout} attribute.
   * <p>
   * Defaults to {@code SECONDS}
   */
  @Parameter
  @Optional(defaultValue = "SECONDS")
  @Placement(tab = ADVANCED_TAB, order = 2)
  @Summary("Time unit to be used in the Timeout configurations")
  private TimeUnit timeoutUnit;

  /**
   * The amount of time that the connection between client and broker will be kept alive without any messages being exchanged.
   * If the keep-alive interval is 0, then the keep-alive mechanism is disabled.
   */
  @Parameter
  @Placement(tab = ADVANCED_TAB, order = 3)
  @Optional(defaultValue = "0")
  @Summary("The maximum amount of time for an inactive connection to be kept alive")
  private int keepAliveInterval;

  /**
   * The unit of time that corresponds to the keepAliveInterval parameter.
   * <p>
   * Defaults to {@code SECONDS}
   */
  @Parameter
  @Placement(tab = ADVANCED_TAB, order = 4)
  @Summary("Time unit to be used as unit for the 'keepAliveInterval' parameter")
  @Optional(defaultValue = "SECONDS")
  private TimeUnit keepAliveIntervalUnit;

  /**
   * The maximum amount of messages that can be unacknowledged at a given time.
   * Setting it to 0 means, there can be unlimited in-flight messages.
   */
  @Parameter
  @Placement(tab = ADVANCED_TAB, order = 5)
  @Optional(defaultValue = "10")
  @Summary("The maximum amount of messages that can be unacknowledged at a given time")
  private int maxInFlight;

  /**
   * If set to true, the session will be cleaned each time the client disconnects from the broker.
   * Subscriptions will not be saved and offline (qos 1 and 2) messages for that client will be lost.
   */
  @Parameter
  @Placement(tab = ADVANCED_TAB, order = 6)
  @Optional(defaultValue = "true")
  @Summary("Whether the session should be cleaned each time the client disconnects from the broker")
  private boolean cleanSession;

  /**
   * @return the connectionTimeout parameter value.
   */
  public int getConnectionTimeout() {
    return connectionTimeout;
  }

  /**
   * @return the connectionTimeout's {@link TimeUnit}.
   */
  public TimeUnit getConnectionTimeoutUnit() {
    return timeoutUnit;
  }

  /**
   * @return the keepAliveInterval parameter value.
   */
  public int getKeepAliveInterval() {
    return keepAliveInterval;
  }

  /**
   * @return the keepAliveInterval's {@link TimeUnit}.
   */
  public TimeUnit getKeepAliveIntervalUnit() {
    return keepAliveIntervalUnit;
  }

  /**
   * @return the maxInFlight parameter value.
   */
  public int getMaxInFlight() {
    return maxInFlight;
  }

  /**
   * @return the cleanSession flag value.
   */
  public boolean getCleanSession() {
    return cleanSession;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof MQTT3ConnectionOptions)) {
      return false;
    }

    MQTT3ConnectionOptions otherConnectionOptions = (MQTT3ConnectionOptions) o;
    return new EqualsBuilder()
        .append(keepAliveInterval, otherConnectionOptions.keepAliveInterval)
        .append(keepAliveIntervalUnit, otherConnectionOptions.keepAliveIntervalUnit)
        .append(maxInFlight, otherConnectionOptions.maxInFlight)
        .append(connectionTimeout, otherConnectionOptions.connectionTimeout)
        .append(timeoutUnit, otherConnectionOptions.timeoutUnit)
        .append(cleanSession, otherConnectionOptions.cleanSession)
        .isEquals();
  }

  @Override
  public int hashCode() {
    return Objects.hash(keepAliveInterval, keepAliveIntervalUnit, maxInFlight, connectionTimeout, timeoutUnit, cleanSession);
  }
}
