/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection;

import com.mulesoft.connectors.mqtt3.internal.exceptions.MQTT3InvalidTopicException;
import com.mulesoft.connectors.mqtt3.internal.exceptions.MQTT3PublishException;
import org.eclipse.paho.client.mqttv3.IMqttActionListener;
import org.eclipse.paho.client.mqttv3.IMqttToken;
import org.mule.runtime.api.connection.ConnectionException;
import org.slf4j.Logger;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import static com.mulesoft.connectors.mqtt3.internal.exceptions.MQTT3ConnectionExceptionResolver.resolveMQTT3ConnectionException;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Represents an action to be executed when a message publication action has been completed.
 */
public class MQTT3PublishActionListener implements IMqttActionListener {

  private static final Logger LOGGER = getLogger(MQTT3PublishActionListener.class);

  private MQTT3Connection connection;
  private CompletableFuture<Integer> future;

  /**
   * Returns a new instance of an {@link MQTT3PublishActionListener}.
   * @param future a {@link CompletableFuture}.
   * @param connection an {@link MQTT3Connection}.
   */
  public MQTT3PublishActionListener(CompletableFuture<Integer> future, MQTT3Connection connection) {
    this.connection = connection;
    this.future = future;
  }

  @Override
  public void onSuccess(IMqttToken iMqttToken) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Successfully published message with MessageId: " + iMqttToken.getMessageId());
    }
    future.complete(iMqttToken.getMessageId());
  }

  @Override
  public void onFailure(IMqttToken iMqttToken, Throwable throwable) {
    LOGGER.error("Exception found on message publish operation: " + throwable.getMessage(), throwable);
    Optional<ConnectionException> connException = resolveMQTT3ConnectionException(throwable, this.connection);
    if (connException.isPresent()) {
      future.completeExceptionally(connException.get());
    } else if (throwable instanceof IllegalArgumentException) {
      future.completeExceptionally(new MQTT3InvalidTopicException(throwable));
    } else {
      future.completeExceptionally(new MQTT3PublishException(throwable));
    }
  }
}
