/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection.provider;

import com.mulesoft.connectors.mqtt3.api.clientId.ClientIDGenerator;
import com.mulesoft.connectors.mqtt3.internal.connection.DefaultMQTT3Connection;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3ConnectionOptions;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3FilePersistenceOptions;
import com.mulesoft.connectors.mqtt3.internal.routing.LWTMessage;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Password;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.sdk.api.annotation.semantics.security.Username;
import org.slf4j.Logger;

import java.util.Objects;

import static org.mule.runtime.api.connection.ConnectionValidationResult.failure;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.slf4j.LoggerFactory.getLogger;


/**
 * Cached Connection provider that creates MQTT3 Connections
 *
 * @since 1.0.0
 */
public abstract class MQTT3ConnectionProvider
    implements CachedConnectionProvider<MQTT3Connection>, Initialisable {

  private static final Logger LOGGER = getLogger(MQTT3ConnectionProvider.class);

  protected static final String DRIVER_FILE_NAME_PATTERN = "(.*)\\.jar";
  protected static final String MQTT_CONNECTION_CLASS_NAME = "org.eclipse.paho.client.mqttv3.MqttAsyncClient";
  protected static final String MQTT_DEPENDENCY = "org.eclipse.paho:org.eclipse.paho.client.mqttv3:1.2.5";

  /**
   * A {@link ClientIDGenerator} that generates a Client ID to identify a connection to the MQTT broker.
   */
  @Parameter
  @Summary("The generator used to create a connection's client id")
  protected ClientIDGenerator clientIDGenerator;

  /**
   * Username to be used when providing authentication credentials to the MQTT broker.
   */
  @Parameter
  @Optional
  @Username
  @Summary("The username used when providing authentication credentials to the MQTT broker")
  protected String username;

  /**
   * Password to be used when providing authentication credentials to the MQTT broker.
   */
  @Parameter
  @Optional
  @Password
  @Summary("The password to be used when providing authentication credentials to the MQTT broker")
  protected String password;

  /**
   * A factory for TLS contexts. A TLS context is configured with a key store and a trust store. Allows to create TLS secured
   * connections.
   */
  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @Placement(tab = "SSL/TLS")
  @DisplayName("TLS Context")
  @Summary("The TLS factory used to create TLS secured connections")
  protected TlsContextFactory tlsContextFactory;

  /**
   * The {@link LWTMessage} to be set.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Placement(tab = "LWT")
  @DisplayName("Last Will and Testament (LWT)")
  @ParameterGroup(name = "Last Will and Testament", showInDsl = true)
  @Summary("The Last Will And Testament message to be set for this client")
  protected LWTMessage lastWillAndTestament;

  /**
   * The provided {@link MQTT3ConnectionOptions}.
   */
  @NullSafe
  @ParameterGroup(name = "Connection Options", showInDsl = true)
  @Summary("A group of advanced connection options")
  protected MQTT3ConnectionOptions connectionOptions;

  /**
   * The provided {@link MQTT3FilePersistenceOptions}.
   */
  @NullSafe
  @ParameterGroup(name = "File Persistence Options", showInDsl = true)
  protected MQTT3FilePersistenceOptions filePersistenceOptions;

  @Override
  public void initialise() throws InitialisationException {
    if (lastWillAndTestament.getBody() == null && lastWillAndTestament.getTopic() != null) {
      throw new InitialisationException(createStaticMessage("Last will and testament message topic provided, but message content was not found."),
                                        this);
    }

    if (lastWillAndTestament.getBody() != null && lastWillAndTestament.getTopic() == null) {
      throw new InitialisationException(createStaticMessage("Last will and testament message content provided, but topic was not found."),
                                        this);
    }
  }

  @Override
  public MQTT3Connection connect() throws ConnectionException {
    LOGGER.debug("Attempting to establish connection to MQTT3 broker.");

    DefaultMQTT3Connection mqtt3Connection =
        new DefaultMQTT3Connection(getUrl(), this.clientIDGenerator.generateClientID(), connectionOptions, filePersistenceOptions,
                                   lastWillAndTestament);

    mqtt3Connection.setUsernamePassword(username, password);

    if (tlsContextFactory != null) {
      mqtt3Connection.setTLSOptions(tlsContextFactory);
    }

    configureMqttConnectOptions(mqtt3Connection);
    // Note: Connection is NOT established here anymore to prevent race conditions
    // The connection will be established when connect() is called explicitly

    return mqtt3Connection;
  }

  /**
   * Sets specific configuration options for an {@link MQTT3Connection}
   * @param mqtt3Connection an instance of {@link MQTT3Connection}
   * @throws ConnectionException
   */
  protected abstract void configureMqttConnectOptions(MQTT3Connection mqtt3Connection) throws ConnectionException;

  /**
   * @return the URL to connect to.
   */
  protected abstract String getUrl();

  @Override
  public void disconnect(MQTT3Connection connection) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("About to disconnect: {}", connection.hashCode());
    }

    connection.disconnect();
  }

  /**
   * Validates the provided {@link MQTT3Connection} by testing its connectivity.
   * Returns a {@link ConnectionValidationResult} indicating whether the connection is valid.
   *
   * @param connection the {@link MQTT3Connection} instance to validate
   * @return a {@link ConnectionValidationResult} representing the outcome of the validation
   */
  @Override
  public ConnectionValidationResult validate(MQTT3Connection connection) {
    LOGGER.debug("Validating Connection");

    if ((connection == null)) {
      return failure("Disconnected", new IllegalStateException("Disconnected"));
    }

    try {
      return connection.testConnectivity() ? success()
          : failure("Disconnected", new IllegalStateException("Unable to establish connection"));
    } catch (Exception e) {
      LOGGER.debug("Exception during connection validation: {}", e.getMessage());
      return failure("Disconnected", e);
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof MQTT3ConnectionProvider)) {
      return false;
    }

    MQTT3ConnectionProvider that = (MQTT3ConnectionProvider) o;

    return new EqualsBuilder().append(this.clientIDGenerator, that.clientIDGenerator)
        .append(username, that.username)
        .append(password, that.password)
        .append(tlsContextFactory, that.tlsContextFactory)
        .append(lastWillAndTestament, that.lastWillAndTestament)
        .append(connectionOptions, that.connectionOptions)
        .append(filePersistenceOptions, that.filePersistenceOptions)
        .isEquals();
  }

  @Override
  public int hashCode() {
    return Objects.hash(this.clientIDGenerator, username, password, tlsContextFactory, lastWillAndTestament, connectionOptions,
                        filePersistenceOptions);
  }

}
