/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection.provider;

import com.mulesoft.connectors.mqtt3.api.FailOverURL;
import com.mulesoft.connectors.mqtt3.api.Protocol;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.ExternalLib;
import org.mule.runtime.extension.api.annotation.dsl.xml.ParameterDsl;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

import java.util.List;
import java.util.Objects;

import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.DRIVER_FILE_NAME_PATTERN;
import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.MQTT_CONNECTION_CLASS_NAME;
import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.MQTT_DEPENDENCY;
import static java.util.Collections.emptyList;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExternalLibraryType.JAR;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;

/**
 * Cached Connection provider that creates MQTT Connections.
 * Allows the user to specify a list of {@link FailOverURL}s that the
 * connection provider can iterate over until a connection is successfully established.
 *
 * @since 1.0.0
 */
@Alias("fail-over")
@DisplayName("MQTT3 Fail-Over Connection")
@ExternalLib(name = "Eclipse Paho MQTT3", description = "Eclipse Paho MQTT3 client library for MQTT v3",
    nameRegexpMatcher = DRIVER_FILE_NAME_PATTERN,
    requiredClassName = MQTT_CONNECTION_CLASS_NAME,
    coordinates = MQTT_DEPENDENCY, type = JAR, optional = true)
public class MQTT3FailOverConnectionProvider extends MQTT3ConnectionProvider {


  /**
   * The list of server URLs that will be used to (attempt to) establish a connection to the broker.
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @ParameterDsl(allowReferences = false)
  @Summary("The list of server URLs to iterate over until a connection to the broker is successfully established")
  private List<FailOverURL> failOverServers = emptyList();

  /**
   * @return the list of {@link FailOverURL}s set for this connection provider.
   */
  public List<FailOverURL> getFailOverServers() {
    return failOverServers;
  }

  @Override
  public void initialise() throws InitialisationException {
    if (failOverServers.isEmpty()) {
      throw new InitialisationException(createStaticMessage("No failOver servers have been specified. At least one mqtt server must be listed."),
                                        this);
    }

    if (failOverServers.stream()
        .filter(failOverURL -> failOverURL.getUrl().startsWith(Protocol.SSL.name())
            || failOverURL.getUrl().startsWith(Protocol.WSS.name()))
        .count() > 0 && (tlsContextFactory == null)) {
      throw new InitialisationException(createStaticMessage("Secure protocols were specified but no tls context was provided for "
          + failOverServers.stream().filter(failOverURL -> failOverURL.getUrl().startsWith(Protocol.SSL.name())
              || failOverURL.getUrl().startsWith(Protocol.WSS.name()))), this);
    }

    initialiseIfNeeded(tlsContextFactory);

    super.initialise();
  }

  @Override
  protected void configureMqttConnectOptions(MQTT3Connection mqtt3Connection) throws ConnectionException {
    String[] failOverServerArray = new String[failOverServers.size()];
    for (int i = 0; i < failOverServers.size(); i++) {
      failOverServerArray[i] = failOverServers.get(i).getUrl();
    }

    mqtt3Connection.setFailOverServers(failOverServerArray);
  }

  @Override
  protected String getUrl() {
    return getFailOverServers().isEmpty() ? "" : getFailOverServers().get(0).getUrl();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof MQTT3FailOverConnectionProvider)) {
      return false;
    }

    MQTT3FailOverConnectionProvider that = (MQTT3FailOverConnectionProvider) o;
    return super.equals(that) && this.failOverServers.equals(that.failOverServers);
  }

  @Override
  public int hashCode() {
    return Objects.hash(clientIDGenerator, username, password, tlsContextFactory, lastWillAndTestament,
                        connectionOptions,
                        failOverServers);
  }
}
