/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection.provider;

import com.mulesoft.connectors.mqtt3.api.Protocol;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.ExternalLib;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.sdk.api.annotation.semantics.connectivity.Host;
import org.mule.sdk.api.annotation.semantics.connectivity.Port;
import org.slf4j.Logger;

import java.util.Objects;

import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.DRIVER_FILE_NAME_PATTERN;
import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.MQTT_CONNECTION_CLASS_NAME;
import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.MQTT_DEPENDENCY;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExternalLibraryType.JAR;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.slf4j.LoggerFactory.getLogger;


/**
 * Cached Connection provider that creates MQTT Connections.
 * Allows the user to specify a protocol, host and port to establish a connection with the broker.
 *
 * @since 1.0.0
 */
@Alias("form")
@DisplayName("MQTT3 Form Connection")
@ExternalLib(name = "Eclipse Paho MQTT3", description = "Eclipse Paho MQTT3 client library for MQTT v3",
    nameRegexpMatcher = DRIVER_FILE_NAME_PATTERN,
    requiredClassName = MQTT_CONNECTION_CLASS_NAME,
    coordinates = MQTT_DEPENDENCY, type = JAR, optional = true)
public class MQTT3FormConnectionProvider extends MQTT3ConnectionProvider {

  private static final Logger LOGGER = getLogger(MQTT3FormConnectionProvider.class);

  /**
   * The protocol to be used when connecting to the broker.
   * <p>
   * Defaults to {@code TCP}
   */
  @Parameter
  @Optional(defaultValue = "TCP")
  @Expression(value = NOT_SUPPORTED)
  @Summary("The protocol to be used when connecting to the broker")
  private Protocol protocol;

  /**
   * The host of the broker to connect to.
   */
  @Host
  @Parameter
  @Example("0.0.0.0")
  @Expression(value = NOT_SUPPORTED)
  @DisplayName("URL")
  @Summary("The host of the broker to connect to")
  private String host;

  /**
   * The broker's port to connect to.
   */
  @Port
  @Parameter
  @Expression(value = NOT_SUPPORTED)
  @Optional(defaultValue = "1883")
  @Summary("The broker's port to connect to")
  private String port;

  private String url;

  @Override
  protected void configureMqttConnectOptions(MQTT3Connection mqtt3Connection) throws ConnectionException {}

  @Override
  protected String getUrl() {
    return protocol + "://" + host + ":"
        + port;
  }

  @Override
  public void initialise() throws InitialisationException {
    try {
      this.url = getUrl();

      if ((this.url.startsWith(Protocol.SSL.name())
          || this.url.startsWith(Protocol.WSS.name())) && (tlsContextFactory == null)) {
        throw new InitialisationException(createStaticMessage("Secure protocol was specified but no tls context was provided "
            + this.url), this);
      }
      initialiseIfNeeded(tlsContextFactory);
    } catch (InitialisationException exception) {
      LOGGER.error("Failed to initialise TLS context. " + exception.getMessage());
      throw new InitialisationException(createStaticMessage("Failed to initialise TLS context. " + exception.getMessage(),
                                                            exception),
                                        this);
    }

    super.initialise();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof MQTT3ConnectionProvider)) {
      return false;
    }

    MQTT3FormConnectionProvider that = (MQTT3FormConnectionProvider) o;
    return super.equals(that) && this.url.equals(that.getUrl());
  }

  @Override
  public int hashCode() {
    return Objects.hash(clientIDGenerator, username, password, tlsContextFactory, lastWillAndTestament,
                        connectionOptions,
                        url);
  }
}
