/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.connection.provider;

import com.mulesoft.connectors.mqtt3.api.Protocol;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.ExternalLib;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.sdk.api.annotation.semantics.connectivity.Url;
import org.slf4j.Logger;

import java.util.Objects;

import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.DRIVER_FILE_NAME_PATTERN;
import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.MQTT_CONNECTION_CLASS_NAME;
import static com.mulesoft.connectors.mqtt3.internal.connection.provider.MQTT3ConnectionProvider.MQTT_DEPENDENCY;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.ExternalLibraryType.JAR;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.slf4j.LoggerFactory.getLogger;


/**
 * Cached Connection provider that creates MQTT Connections.
 * Allows the user to specify a connection URL for the broker.
 *
 * @since 1.0.0
 */

@DisplayName("MQTT3 URL Connection")
@ExternalLib(name = "Eclipse Paho MQTT3", description = "Eclipse Paho MQTT3 client library for MQTT v3",
    nameRegexpMatcher = DRIVER_FILE_NAME_PATTERN,
    requiredClassName = MQTT_CONNECTION_CLASS_NAME,
    coordinates = MQTT_DEPENDENCY, type = JAR, optional = true)
public class MQTT3UrlConnectionProvider extends MQTT3ConnectionProvider {

  private static final Logger LOGGER = getLogger(MQTT3UrlConnectionProvider.class);

  /**
   * The url that represents the broker to connect to.
   */
  @Url
  @Parameter
  @Example("tcp://0.0.0.0:1883")
  @Expression(value = NOT_SUPPORTED)
  @DisplayName("URL")
  @Summary("The url to connect to")
  private String url;

  @Override
  protected void configureMqttConnectOptions(MQTT3Connection mqtt3Connection) throws ConnectionException {}

  @Override
  protected String getUrl() {
    return url;
  }

  @Override
  public void initialise() throws InitialisationException {
    try {
      if ((url.startsWith(Protocol.SSL.name())
          || url.startsWith(Protocol.WSS.name())) && (tlsContextFactory == null)) {
        throw new InitialisationException(createStaticMessage("Secure protocol was specified but no tls context was provided "
            + url), this);
      }
      initialiseIfNeeded(tlsContextFactory);
    } catch (InitialisationException exception) {
      LOGGER.error("Failed to initialise TLS context. " + exception.getMessage(), exception);
      throw new InitialisationException(createStaticMessage("Failed to initialise TLS context. " + exception.getMessage(),
                                                            exception),
                                        this);
    }

    super.initialise();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (!(o instanceof MQTT3ConnectionProvider)) {
      return false;
    }

    MQTT3UrlConnectionProvider that = (MQTT3UrlConnectionProvider) o;
    return super.equals(that) && this.url.equals(that.url);
  }

  @Override
  public int hashCode() {
    return Objects.hash(clientIDGenerator, username, password, tlsContextFactory, lastWillAndTestament,
                        connectionOptions,
                        url);
  }
}
