/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.exceptions;

import org.eclipse.paho.client.mqttv3.MqttException;
import org.mule.runtime.api.connection.ConnectionException;

import java.util.List;
import java.util.Optional;

import static java.util.Arrays.asList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.eclipse.paho.client.mqttv3.MqttException.REASON_CODE_CLIENT_CLOSED;
import static org.eclipse.paho.client.mqttv3.MqttException.REASON_CODE_CLIENT_NOT_CONNECTED;
import static org.eclipse.paho.client.mqttv3.MqttException.REASON_CODE_CONNECTION_LOST;
import static org.eclipse.paho.client.mqttv3.MqttException.REASON_CODE_SERVER_CONNECT_ERROR;

/**
 * Resolves whether a specific exception's root cause is connectivity related by analysing
 * the error code provided by the driver.
 */
public class MQTT3ConnectionExceptionResolver {

  /**
   * A list of error codes that are set by the driver when a connectivity error is thrown.
   */
  private static final List<Short> connectionErrorCodes =
      asList(REASON_CODE_CLIENT_NOT_CONNECTED, REASON_CODE_SERVER_CONNECT_ERROR, REASON_CODE_CONNECTION_LOST,
             REASON_CODE_CLIENT_CLOSED);

  /**
   * Resolves whether a specific exception's root cause is connectivity related.
   * @param throwable the instance of {@link Throwable} to be analysed
   * @param conn the connection object associated to the exception.
   * @return an {@link Optional} of {@link ConnectionException} if a connectivity related error is detected. Otherwise, returns an empty {@link Optional}.
   */
  public static Optional<ConnectionException> resolveMQTT3ConnectionException(Throwable throwable, Object conn) {
    Optional<ConnectionException> exception = empty();
    if (throwable instanceof MqttException) {
      MqttException mqttException = (MqttException) throwable;
      if (connectionErrorCodes.contains((short) mqttException.getReasonCode())) {
        exception = of(new ConnectionException(throwable, conn));
      }
    }

    return exception;
  }
}
