/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.operation;

import com.mulesoft.connectors.mqtt3.api.QoS;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import com.mulesoft.connectors.mqtt3.internal.exceptions.MQTT3PublishErrorTypes;
import com.mulesoft.connectors.mqtt3.internal.routing.MQTT3Message;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.slf4j.Logger;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

import static org.mule.runtime.core.api.util.IOUtils.toByteArray;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * MQTT3 Connector operations
 */
public class MQTT3Operations {

  private static final Logger LOGGER = getLogger(MQTT3Operations.class);

  /**
   * Operation that allows the user to publish a single {@link MQTT3Message} to a given exchange
   *
   * @param topic the topic to which the message must be published
   * @param connection the current {@link MQTT3Connection}
   * @param message the content of the message to be published
   * @param qos the {@link QoS} with which the message will be published
   * @param isRetained whether the message should be retained by the broker
   */
  @Throws(MQTT3PublishErrorTypes.class)
  @MediaType(value = MediaType.ANY)
  public void publish(String topic,
                      @Connection MQTT3Connection connection,
                      @Content InputStream message,
                      @Optional(defaultValue = "AT_LEAST_ONCE") QoS qos,
                      boolean isRetained,
                      CompletionCallback<Void, Void> completionCallback) {
    try {

      CompletableFuture<Integer> future =
          connection.publish(topic, toByteArray(message), qos.getValue(), isRetained);
      future.whenComplete((messageId, mqttError) -> {
        if (mqttError == null) {
          LOGGER.debug("Message with id " + messageId + " successfully published");
          completionCallback.success(null);
        } else {
          completionCallback.error(mqttError);
        }
      });

    } catch (Throwable mqttException) {
      completionCallback.error(mqttException);
    }
  }
}
