/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.routing;

import com.mulesoft.connectors.mqtt3.api.Topic;
import org.mule.runtime.api.util.Pair;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;

import static java.util.stream.Collectors.toList;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Maps subscription {@link Topic}s to a set of {@link MQTT3MessageHandler}s. Uses a {@link MQTT3TopicMatcher} to route
 * the received {@link MQTT3Message}s to the corresponding handlers.
 */
public class MQTT3TopicRouter {

  private static final Logger LOGGER = getLogger(MQTT3TopicRouter.class);

  private final MQTT3TopicMatcher topicMatcher;
  /**
   * Maps each {@link Topic} to all its registered {@link MQTT3MessageHandler}s.
   */
  private ConcurrentHashMap<String, Pair<Topic, List<MQTT3MessageHandler>>> topicCallbacksRegistry;

  /**
   * Returns a new instance of {@link MQTT3TopicRouter}
   * @param topicMatcher the {@link MQTT3TopicMatcher} to use to match incoming messages to the corresponding {@link Topic}s.
   */
  public MQTT3TopicRouter(MQTT3TopicMatcher topicMatcher) {
    this.topicMatcher = topicMatcher;
    this.topicCallbacksRegistry = new ConcurrentHashMap<>();
  }

  /**
   * Receives a list of topics and the message handler to be invoked when messages for those topics are received.
   * If the topic is being registered for the first time, a list of handlers is created and associated to that topic,
   * and the messageHandler is added to the list.
   * If the topic already has registered handlers, it adds the messageHandler to the list.
   * Returns the list of the topics which are being registered for the first time.
   */
  public synchronized List<Topic> registerCallbackForTopics(List<Topic> topics, MQTT3MessageHandler messageHandler) {
    List<Topic> newTopics = new ArrayList<>();
    for (Topic topic : topics) {
      if (topicCallbacksRegistry.containsKey(topic.getTopicFilter())) {
        LOGGER.debug("Topic {} already subscribed to with qos {}", topic.getTopicFilter(),
                     topic.getQos().getValue());
        topicCallbacksRegistry.get(topic.getTopicFilter()).getSecond().add(messageHandler);
      } else {
        newTopics.add(topic);
        List callbackList = new ArrayList();
        callbackList.add(messageHandler);
        topicCallbacksRegistry.put(topic.getTopicFilter(), new Pair<>(topic, callbackList));
      }
    }
    return newTopics;
  }

  /**
   * Receives a list of topics and the message handler to be removed from the registry.
   * Removes the provided handler from the list of handlers, for each of the provided topics.
   * Returns the list of topics which no longer have any associated handlers.
   */
  public synchronized List<Topic> deregisterCallbackForTopics(List<Topic> topics,
                                                              MQTT3MessageHandler messageHandler) {
    List<Topic> deletedTopics = new ArrayList<>();
    for (Topic topic : topics) {
      if (topicCallbacksRegistry.containsKey(topic.getTopicFilter())) {
        List<MQTT3MessageHandler> callbacksList =
            topicCallbacksRegistry.get(topic.getTopicFilter()).getSecond();
        callbacksList.remove(messageHandler);
        if (callbacksList.isEmpty()) {
          topicCallbacksRegistry.remove(topic.getTopicFilter());
          deletedTopics.add(topic);
        }
      }
    }
    return deletedTopics;
  }

  /**
   * @return a list of all the distinct registered topic filters.
   */
  public List<Topic> getDistinctTopicFilters() {
    return topicCallbacksRegistry.values().stream().map(Pair::getFirst).collect(toList());
  }

  /**
   * Invokes each registered {@link MQTT3MessageHandler} associated to a {@link Topic}
   * that matches the incoming {@link MQTT3Message}'s destination topic.
   * @param mqttMessage the incoming {@link MQTT3Message}
   */
  public void handleMessageArrived(MQTT3Message mqttMessage) {
    Predicate<Map.Entry<String, Pair<Topic, List<MQTT3MessageHandler>>>> topicMatchesFilter =
        entry -> topicMatcher.topicMatches(entry.getKey(), mqttMessage.getTopic());

    topicCallbacksRegistry.entrySet().stream().filter(topicMatchesFilter)
        .forEach(pair -> pair.getValue().getSecond().stream().forEach(callback -> {
          callback.onMessageArrived(mqttMessage);
        }));
  }

}
