/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectors.mqtt3.internal.source;

import com.mulesoft.connectors.mqtt3.api.MQTT3MessageAttributes;
import com.mulesoft.connectors.mqtt3.api.Topic;
import com.mulesoft.connectors.mqtt3.internal.connection.MQTT3Connection;
import com.mulesoft.connectors.mqtt3.internal.routing.MQTT3ForwardingMessageHandler;
import com.mulesoft.connectors.mqtt3.internal.routing.MQTT3Message;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.source.ClusterSupport;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.Source;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;
import org.slf4j.Logger;

import java.util.List;

import static org.mule.runtime.core.api.util.ExceptionUtils.extractConnectionException;
import static org.mule.runtime.extension.api.annotation.param.MediaType.ANY;
import static org.mule.runtime.extension.api.annotation.source.SourceClusterSupport.DEFAULT_PRIMARY_NODE_ONLY;
import static org.slf4j.LoggerFactory.getLogger;

@MediaType(value = ANY)
@Alias("listener")
@DisplayName("On New Message")
@ClusterSupport(value = DEFAULT_PRIMARY_NODE_ONLY)
public class MQTT3MessageListener extends Source<byte[], MQTT3MessageAttributes> {

  private static final Logger LOGGER = getLogger(MQTT3MessageListener.class);

  /**
   * The list of {@link Topic}s that this listener will subscribe to.
   */
  @Parameter
  private List<Topic> topics;

  @Connection
  private ConnectionProvider<MQTT3Connection> connectionProvider;

  private MQTT3Connection connection;
  private MQTT3ForwardingMessageHandler messageHandler;


  @Override
  public void onStart(SourceCallback<byte[], MQTT3MessageAttributes> sourceCallback) {
    try {
      LOGGER.debug("Starting MQTT3 Source");
      connection = connectionProvider.connect();
      connection.addSourceCallbackToConnectionLostHandler(sourceCallback);
      messageHandler = new MQTT3ForwardingMessageHandler(message -> sourceCallback.handle(buildResult(message)));
      connection.registerMessageHandler(topics, messageHandler);
      connection.connect();
      connection.subscribeToTopics(topics);
    } catch (Throwable e) {
      if (extractConnectionException(e).isPresent()) {
        throw new MuleRuntimeException(extractConnectionException(e).get());
      } else {
        throw new MuleRuntimeException(e);
      }
    }
  }

  @Override
  public void onStop() {
    if (connection == null) {
      return;
    }

    try {
      connection.unsubscribeListenerFromTopics(topics, this.messageHandler);
      connectionProvider.disconnect(connection);
    } catch (Exception e) {
      LOGGER.error("Error occurred unsubscribing listeners from topics:" + topics.toString() + " " + e.getMessage(), e);
    }
  }

  private Result<byte[], MQTT3MessageAttributes> buildResult(MQTT3Message mqttMessage) {
    return Result.<byte[], MQTT3MessageAttributes>builder()
        .attributes(MQTT3MessageAttributes.Builder.newInstance()
            .withTopic(mqttMessage.getTopic())
            .withMessageId(mqttMessage.getId())
            .withDuplicate(mqttMessage.getIsDuplicate())
            .withRetained(mqttMessage.getIsRetained())
            .withQoS(mqttMessage.getQoS())
            .build())
        .output(mqttMessage.getContent())
        .build();
  }
}
